/* =======================================================================
 *   (c) Copyright Hewlett-Packard Development Company, L.P., 2008
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of version 2 the GNU General Public License as
 *   published by the Free Software Foundation.
 *   
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *   
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * =======================================================================
 */

/*
 * Copyright (c) 2002-2004 Hewlett-Packard Development Company, L.P.
 */

/* adding new commands to fsadm...
 *
 * make a C module called "<action>.c"
 *
 * make all functions static except "<action>_main(int, char **)"
 * and "<action>_usage(void)"
 *
 * put the declaration for <action>_main and <action>_usage into actions.h
 *
 * add "action" struct entry to the action_list[] array in this file
 *
 * any functions you think might be "common" and useful to several modules
 * should be put in common.c with declarations of the public interface in 
 * common.h.
 */

/* includes */
#include        <stdio.h>
#include        <errno.h>
#include        <stdlib.h>
#include        <string.h>
#include        <fcntl.h>
#include        <unistd.h>
#include        <mntent.h>
#include        <signal.h>
#include        <sys/wait.h>
#include        <sys/stat.h>
#include        <sys/ustat.h>
#include        <sys/types.h>
#include        <sys/mount.h>
#include        <sys/fs.h>
#include        <sys/vfs.h>
#include        <sys/inode.h>
#include        <sys/ino.h>
#include        <sys/fstyp.h>
#include        <sys/libfs.h>
#include        <sys/statvfs.h>
#include        <locale.h>

#include        "common.h"
#include        "actions.h"

/* variables, structs, etc */
char    *fstype = NULL; /* Type of file system (hfs) */
char    *action = NULL; /* action to be performed */
nl_catd catd;           /* Natural language catalog */
int     full_usage = TRUE;

action_t action_list[] = {
    /* { "sample", sample_main, sample_usage }, */
    { "chfs", chfs_main, chfs_usage },
    { "chio", chio_main, chio_usage },
    { "addvol", addvol_main, addvol_usage },
    { "rmvol", rmvol_main, rmvol_usage },
    { "list", list_main, list_usage },
    { "extend", extend_main, extend_usage },
    { "getattr", getattr_main, getattr_usage },
    { "mvlog", mvlog_main, mvlog_usage },
    { "info", info_main, info_usage },
    { "migrate", migrate_main, migrate_usage },
    { "autotune", autotune_main, autotune_usage },
    { "scan", scan_main, scan_usage },
    { "defrag", defrag_main, defrag_usage },
    { "rmfs", rmfs_main, rmfs_usage },
    { "balance", balance_main, balance_usage },
    { "multi", multi_main, multi_usage },
    { "rename", rename_main, rename_usage },
    { "prealloc", prealloc_main, prealloc_usage },
    { "snap", snap_main, snap_usage },
    { "promote", promote_main, promote_usage },
    { "demote", demote_main, demote_usage },
    { "clu_helpers", helper_main, helper_usage },
    { NULL, NULL, NULL }
};

/* static functions */
static void fsadm_usage( void );		  /* usage routine prints ALL
						   * fsadm module usages */
static int actionhandler( int argc, char **argv); /* parse the fsadm_advfs
						   * command line to extract
						   * action then branch to
						   * action handler */

/*
 *      NAME:
 *              main() - fsadm_advfs(1M)
 *
 *      DESCRIPTION:
 *              fsadm_advfs driver
 *
 *      ARGUMENTS:
 *              <argc> - number of command line arguments
 *              <argv> - command line arguments
 *
 *      RETURN VALUES:
 *              success - 0
 *              failure - non-zero
 */

int
main( int argc, char **argv )
{
        int	error=0;
        
#if defined(NLS) || defined(NLS16)
        if (!setlocale(LC_ALL, "")) {
            fputs(_errlocale(name), stderr);
            putenv("LANG=");
        }
        catd = catopen(MF_FSADM_ADVFS, NL_CAT_LOCALE);
#endif

        /* ignore SIGSYS -- it is generated by a kernel that does not
         * contain AdvFS when an advfs_syscall is attempted.  Without
         * this SIG_IGN (ignore) the command will segfault
         */
        signal( SIGSYS, SIG_IGN );
        
        /* actionhandler() will branch to the appropriate action routine */
        if ( (error=actionhandler(argc,argv)) ) {
                return(error);
        }

        return(error);
}

/*
 *      NAME:
 *              actionhandler()
 *
 *      DESCRIPTION:
 *                i) parses out the fsadm_advfs command.
 *               ii) branches to specific module for command handling.
 *
 *      ARGUMENTS:
 *              <argc> - number of command line arguments
 *              <argv> - command line arguments
 *
 *      RETURN VALUES:
 *              success - 0
 *              failure - 1 or errno
 */

static int
actionhandler( int argc, char **argv )
{
        char    **tmp_argv = NULL;
        int     tmp_argc = 0;
        int     i = 0, j = 0;
        int     fsflag = 0, key = 0;
        
        /* store a temp pointer to the arg vector starting at the first
         * argument (i.e. exclude the command name) */
        tmp_argv = (argv + 1);
        tmp_argc = argc - 1;

        for(; tmp_argc > 0; tmp_argc--, tmp_argv++) {
                /* check to see if the argument begins with a "-F", which
                 * would make it the only option flag this parsing accepts */
                if(strstr(*tmp_argv, "-F") == *tmp_argv) {
                        /* make sure we haven't already seen the "-F" option */
                        if(fsflag) {
                                fsadm_usage();
                                return(1);
                        }

                        /* we can now set the fsflag */
                        fsflag++;

                        /* parse out the fstype if it is in this argument */
                        if(strlen(*tmp_argv) != 2) {
                                fstype = strdup((*tmp_argv)+2);
                                fsflag++; /* fsflag == 2 when fstype has been set */
                        }
                }
                /* since the argument did not start with "-F", this is not a
                 * valid option flag at this point, verify the argument is NOT
                 * another option */
                else if( strchr( *tmp_argv, '-' ) != *tmp_argv ) {

			/* We haven't set the fstype yet */
                        if(fsflag == 1) {
                                fstype = strdup(*tmp_argv);
                                fsflag++; /* fsflag == 2 when fstype has been set */
                        }

			/*
			 * Loop through the action list to see if this is an
			 * action.  If it is, move on.  If not, assume "info"
			 */
			else {
			    i = 0;
			    while( action_list[i].action_name != NULL ) {
				if( !strcmp( action_list[i].action_name,
					     *tmp_argv ) ) {
				    action = strdup( *tmp_argv );
				    key = i;
				    break;
				}
				i++;
			    }
			    break;
			}
                }
                /* if we reach here, the argument is neither "-F", an fstype,
                 * or an action.  Thus we break */
                else {
                        break;
                }
        }
        
        /* after the initial parsing, make sure that fstype is
         * non-NULL and that it is correctly set to "advfs", OR that
         * we didn't have an unqualified "-F" flag */
        if((fstype != NULL && strcmp(fstype, MNTTYPE_ADVFS)) ||
           fsflag == 1) {
                fsadm_usage();
                return(1);
        }

        /* if fstype IS NULL, it must be set to the default */
        if(fstype == NULL) {
                fstype = strdup("advfs");
        }

	/* If action hasn't been set, assume it's "info"
	 *
	 * Check info's arguments to make sure they're valid.  If not
	 * call fsadm_usage for FULL usage
	 */
	if( action == NULL ) {

	    action = strdup( "info" );
	    tmp_argv--;
	    tmp_argc++;
	    *tmp_argv = action;

	    /* Check arguments */
	    i = j = 0;
	    if( check_info_args( tmp_argc, tmp_argv, &i, &j ) == NULL ) {
		fsadm_usage();
		return 1;
	    }

	    /* Since we assuming "info" go ahead and call info_main direcly */
	    return( info_main( tmp_argc, tmp_argv ) );

	} else {

	    /* Call out the appropriate module */
	    return( action_list[key].action_main( tmp_argc, tmp_argv ) );
	}
 
        fsadm_usage();
        return(1);
}

void
fsadm_usage( void )
{
    int i=0;

    info_usage();

    /* We just printed the CUSTOM_USAGE message don't need anymore. */
    full_usage = FALSE;

    /* Loop through the usage routines for all the fsadm commands */
    for( i=0; action_list[i].action_name != NULL; i++ ) {

	/* If this is info, skip it, we already did that one */
	if( !strcmp( "info", action_list[i].action_name ) ) {
	    continue;
	}

        /* clu_helpers is a hidden subcommand intended only for 
         * use with system management/cluster creation scripts.  Don't
         * print out the usage.
         */
	if ( !strcmp( "clu_helpers", action_list[i].action_name ) ) {
	    continue;
	}

	action_list[i].action_usage();
    }
}
