/*
 * hfsck - tool for checking and repairing the integrity of HFS volumes
 * Copyright (C) 1996, 1997 Robert Leslie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: main.c,v 1.7 1997/09/14 19:59:11 rob Exp $
 */

# include <stdio.h>
# include <string.h>
# include <unistd.h>
# include <sys/types.h>
# include <sys/stat.h>
# include <fcntl.h>
# include <errno.h>

# include "hfsck.h"
# include "suid.h"
# include "version.h"

int options;

extern int optind;

/*
 * NAME:	usage()
 * DESCRIPTION:	display usage message
 */
static
int usage(char *argv[])
{
  fprintf(stderr, "Usage: %s [-v] [-n] [-a] device-path\n", argv[0]);

  return 1;
}

/*
 * NAME:	main()
 * DESCRIPTION:	program entry
 */
int main(int argc, char *argv[])
{
  char *path;
  int pnum, result;
  hfsvol vol;

  suid_init(0);

  if (argc == 2)
    {
      if (strcmp(argv[1], "--version") == 0)
	{
	  printf("%s - %s\n", hfsutils_version, hfsutils_copyright);
	  printf("`%s --license' for licensing information.\n", argv[0]);
	  return 0;
	}
      else if (strcmp(argv[1], "--license") == 0)
	{
	  printf("\n%s", hfsutils_license);
	  return 0;
	}
    }

  options = HFSCK_REPAIR;

  while (1)
    {
      int opt;

      opt = getopt(argc, argv, "vna");
      if (opt == EOF)
	break;

      switch (opt)
	{
	case '?':
	  return usage(argv);

	case 'v':
	  options |= HFSCK_VERBOSE;
	  break;

	case 'n':
	  options &= ~HFSCK_REPAIR;
	  break;

	case 'a':
	  options |= HFSCK_YES;
	  break;
	}
    }

  if (argc - optind != 1)
    return usage(argv);

  path = argv[optind];
  pnum = 1;

  v_init(&vol, pnum);

  if (REPAIR)
    {
      result = v_open(&vol, path, HFS_MODE_RDWR);
      if (result == -1)
	{
	  vol.flags |= HFS_VOL_READONLY;
	  result = v_open(&vol, path, HFS_MODE_RDONLY);
	}
    }

  if (result == -1)
    {
      perror(path);
      return 1;
    }

  if (REPAIR && (vol.flags & HFS_VOL_READONLY))
    {
      fprintf(stderr, "%s: warning, %s not writable; cannot repair\n",
	      argv[0], path);

      options &= ~HFSCK_REPAIR;
    }

  if (l_structure(&vol) == -1 ||
      l_readmdb(&vol) == -1)
    {
      perror(path);
      v_close(&vol);
      return 1;
    }

  result = hfsck(&vol);

  if (result == 0 && v_flush(&vol, 1) == -1)
    {
      perror("flushing volume");
      v_close(&vol);
      return 1;
    }

  if (v_close(&vol) == -1)
    {
      perror("closing volume");
      return 1;
    }

  return result;
}
