# include <stdio.h>
# include <string.h>
# include <stdlib.h>
# include <fcntl.h>

# include "hfs.h"
# include "hcwd.h"
# include "hfsutil.h"
# include "glob.h"
# include "hattrib.h"

#ifdef __BEOS__

int external_func(int argc, char *argv[])
{
  return hattrib_main(argc, argv);
}

#endif // __BEOS__

/*
 * NAME:	usage()
 * DESCRIPTION:	display usage message
 */
static
int usage(char *argv[])
{
  fprintf(stderr,
	  "Usage: %s [-t TYPE] [-c CREA] [-|+i] [-|+l] hfs-path [...]\n",
	  argv[0]);

  return 1;
}

/*
 * NAME:	hattrib->main()
 * DESCRIPTION:	implement hattrib command
 */
int hattrib_main(int argc, char *argv[])
{
  char *type = 0, *crea = 0;
  int invis = 0, lock = 0;
  hfsvol *vol;
  int fargc;
  char **fargv;
  int i, result = 0;

  for (i = 1; i < argc; ++i)
    {
      switch (argv[i][0])
	{
	case '-':
	  switch (argv[i][1])
	    {
	    case 't':
	      type = argv[++i];

	      if (type == 0)
		return usage(argv);

	      if (strlen(type) != 4)
		{
		  fprintf(stderr, "%s: file type must be 4 characters\n",
			  argv[0]);
		  return 1;
		}
	      continue;

	    case 'c':
	      crea = argv[++i];

	      if (crea == 0)
		return usage(argv);

	      if (strlen(crea) != 4)
		{
		  fprintf(stderr, "%s: file creator must be 4 characters\n",
			  argv[0]);
		  return 1;
		}
	      continue;

	    case 'i':
	      invis = -1;
	      continue;

	    case 'l':
	      lock = -1;
	      continue;

	    default:
	      return usage(argv);
	    }
	  break;

	case '+':
	  switch (argv[i][1])
	    {
	    case 'i':
	      invis = 1;
	      continue;

	    case 'l':
	      lock = 1;
	      continue;

	    default:
	      return usage(argv);
	    }
	  break;
	}

      break;
    }

  if (argc - i == 0)
    return usage(argv);

  if (i == 1)
    {
      fprintf(stderr, "%s: no attributes specified\n", argv[0]);
      return 1;
    }

  vol = hfs_remount(hcwd_getvol(-1), O_RDWR);
  if (vol == 0)
    return 1;

  fargv = hfs_glob(vol, argc - i, &argv[i], &fargc);
  if (fargv == 0)
    {
      fprintf(stderr, "%s: out of memory\n", argv[0]);
      result = 1;
    }
  else
    {
      for (i = 0; i < fargc; ++i)
	{
	  hfsdirent ent;

	  if (hfs_stat(vol, fargv[i], &ent) < 0)
	    {
	      hfs_perrorp(fargv[i]);
	      result = 1;
	    }
	  else
	    {
	      if (type)
		memcpy(ent.type, type, 4);
	      if (crea)
		memcpy(ent.creator, crea, 4);

	      if (invis < 0)
		ent.fdflags &= ~HFS_FNDR_ISINVISIBLE;
	      else if (invis > 0)
		ent.fdflags |= HFS_FNDR_ISINVISIBLE;

	      if (lock < 0)
		ent.flags &= ~HFS_ISLOCKED;
	      else if (lock > 0)
		ent.flags |= HFS_ISLOCKED;

	      if (hfs_setattr(vol, fargv[i], &ent) < 0)
		{
		  hfs_perrorp(fargv[i]);
		  result = 1;
		}
	    }
	}
    }

  if (hfs_umount(vol) < 0 && result == 0)
    {
      hfs_perror("Error closing HFS volume");
      result = 1;
    }

  if (fargv)
    free(fargv);

  return result;
}
