/*
 * hfsutils - tools for reading and writing Macintosh HFS volumes
 * Copyright (C) 1996 Robert Leslie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

# include <stdlib.h>
# include <unistd.h>
# include <fcntl.h>
# include <errno.h>
# include <string.h>
# include <netinet/in.h>
# include <time.h>
# include <ctype.h>

# include "hfs.h"

# ifdef DMALLOC
# include <dmalloc.h>
# else

# define SIZE(type, n)		((size_t) (sizeof(type) * (n)))
# define ALLOC(type, n)		((type *) malloc(SIZE(type, n)))
# define ALLOCX(type, n)	((n) ? ALLOC(type, n) : (type *) 0)
# define FREE(ptr)		((void) ((ptr) ? free((void *) ptr) : 0))

# define REALLOC(ptr, type, n)  \
    ((type *) ((ptr) ? realloc(ptr, SIZE(type, n)) : malloc(SIZE(type, n))))
# define REALLOCX(ptr, type, n)  \
    ((n) ? REALLOC(type, n) : (FREE(ptr), (type *) 0))

# endif

typedef char	byte;		/* MUST be 1 byte wide */
typedef short	word;		/* MUST be 2 bytes wide */
typedef long	lword;		/* MUST be 4 bytes wide */

char *hfs_error = "no error";	/* static error string */

static hfsvol *mounts = 0;	/* linked list of mounted volumes */

# define ERROR(code, str)	(hfs_error = (str), errno = (code))

/* Data Marshaling Routines ================================================ */

/*
 * NAME:	getmb()
 * DESCRIPTION:	marshal 1 byte into local host format
 */
static
byte getmb(char *ptr)
{
  return *(byte *) ptr;
}

/*
 * NAME:	getmw()
 * DESCRIPTION:	marshal 2 bytes into local host format
 */
static
word getmw(char *ptr)
{
  word data;

  memcpy(&data, ptr, sizeof(data));
  return ntohs(data);
}

/*
 * NAME:	getml()
 * DESCRIPTION:	marshal 4 bytes into local host format
 */
static
lword getml(char *ptr)
{
  lword data;

  memcpy(&data, ptr, sizeof(data));
  return ntohl(data);
}

/*
 * NAME:	putmb()
 * DESCRIPTION:	marshal 1 byte out to Macintosh (big-endian) format
 */
static
void putmb(char *ptr, byte data)
{
  *(byte *) ptr = data;
}

/*
 * NAME:	putmw()
 * DESCRIPTION:	marshal 2 bytes out to Macintosh (big-endian) format
 */
static
void putmw(char *ptr, word data)
{
  data = htons(data);
  memcpy(ptr, &data, sizeof(data));
}

/*
 * NAME:	putml()
 * DESCRIPTION:	marshal 4 bytes out to Macintosh (big-endian) format
 */
static
void putml(char *ptr, lword data)
{
  data = htonl(data);
  memcpy(ptr, &data, sizeof(data));
}

/*
 * NAME:	fetchb()
 * DESCRIPTION:	incrementally retrieve a byte of data
 */
static
void fetchb(char **ptr, byte *dest)
{
  *dest = getmb(*ptr);
  *ptr += sizeof(byte);
}

/*
 * NAME:	fetchw()
 * DESCRIPTION:	incrementally retrieve a word of data
 */
static
void fetchw(char **ptr, word *dest)
{
  *dest = getmw(*ptr);
  *ptr += sizeof(word);
}

/*
 * NAME:	fetchl()
 * DESCRIPTION:	incrementally retrieve a long word of data
 */
static
void fetchl(char **ptr, lword *dest)
{
  *dest = getml(*ptr);
  *ptr += sizeof(lword);
}

/*
 * NAME:	fetchs()
 * DESCRIPTION:	incrementally retrieve a string
 */
static
void fetchs(char **ptr, char *dest, int size)
{
  unsigned char len;

  fetchb(ptr, &len);
  if (len > 0 && len < size)
    memcpy(dest, *ptr, len);
  else
    len = 0;

  dest[len] = 0;

  *ptr += size - 1;
}

/*
 * NAME:	storeb()
 * DESCRIPTION:	incrementally store a byte of data
 */
static
void storeb(char **ptr, byte data)
{
  putmb(*ptr, data);
  *ptr += sizeof(byte);
}

/*
 * NAME:	storew()
 * DESCRIPTION:	incrementally store a word of data
 */
static
void storew(char **ptr, word data)
{
  putmw(*ptr, data);
  *ptr += sizeof(word);
}

/*
 * NAME:	storel()
 * DESCRIPTION:	incrementally store a long word of data
 */
static
void storel(char **ptr, lword data)
{
  putml(*ptr, data);
  *ptr += sizeof(lword);
}

/*
 * NAME:	stores()
 * DESCRIPTION:	incrementally store a string
 */
static
void stores(char **ptr, char *src, int size)
{
  unsigned char len;

  len = strlen(src);
  if (len > size)
    len = 0;

  storeb(ptr, len);
  memcpy(*ptr, src, len);

  *ptr += size - 1;
}

/* Macintosh/UNIX Conversion Routines ====================================== */

/*
 * NAME:	utomtime()
 * DESCRIPTION:	convert UNIX to Macintosh time
 */
static
unsigned long utomtime(time_t secs)
{
  return secs + 2082844800;
}

/*
 * NAME:	mtoutime()
 * DESCRIPTION:	convert Macintosh to UNIX time
 */
static
time_t mtoutime(unsigned long secs)
{
  return secs - 2082844800;
}

/*
 * NAME:	mtoutype()
 * DESCRIPTION:	convert Macintosh OSType to UNIX char string
 */
static
void mtoutype(long ostype, char *str)
{
  str[0] = (ostype & 0xff000000) >> 24;
  str[1] = (ostype & 0x00ff0000) >> 16;
  str[2] = (ostype & 0x0000ff00) >>  8;
  str[3] = (ostype & 0x000000ff) >>  0;

  str[4] = 0;
}

/* Low-Level Block Routines ================================================ */

/*
 * NAME:	readlblock()
 * DESCRIPTION:	read a logical block (sector) from a volume
 */
static
int readlblock(hfsvol *vol, unsigned long num, block *ptr)
{
  int bytes;

  if (lseek(vol->fd, (vol->vstart + num) * HFS_BLOCKSZ, SEEK_SET) < 0)
    {
      ERROR(errno, "error seeking device");
      return -1;
    }

  bytes = read(vol->fd, ptr, HFS_BLOCKSZ);
  if (bytes < 0)
    {
      ERROR(errno, "error reading from device");
      return -1;
    }
  else if (bytes == 0)
    {
      ERROR(EIO, "read past end of volume");
      return -1;
    }
  else if (bytes != HFS_BLOCKSZ)
    {
      ERROR(EIO, "read truncated block");
      return -1;
    }

  return 0;
}

/*
 * NAME:	writelblock()
 * DESCRIPTION:	write a logical block (sector) to a volume
 */
static
int writelblock(hfsvol *vol, unsigned long num, block *ptr)
{
  int bytes;

  if (lseek(vol->fd, (vol->vstart + num) * HFS_BLOCKSZ, SEEK_SET) < 0)
    {
      ERROR(errno, "error seeking device");
      return -1;
    }

  bytes = write(vol->fd, ptr, HFS_BLOCKSZ);
  if (bytes < 0)
    {
      ERROR(errno, "error writing to device");
      return -1;
    }
  else if (bytes != HFS_BLOCKSZ)
    {
      ERROR(EIO, "wrote truncated block");
      return -1;
    }

  vol->mdb.drLsMod = utomtime(time(0));
  ++vol->mdb.drWrCnt;

  vol->flags |= HFS_TOUCHED | HFS_UPDATE_MDB;

  return 0;
}

/*
 * NAME:	readablock()
 * DESCRIPTION:	read a block from an allocation block from a volume
 */
static
int readablock(hfsvol *vol,
	       unsigned int anum, unsigned int index, block *ptr)
{
  /* verify the allocation block exists and is marked as in-use */

  if (anum >= vol->mdb.drNmAlBlks)
    {
      ERROR(EIO, "read nonexistent block");
      return -1;
    }

  if (! HFS_BMTST(vol->vbm, anum))
    {
      ERROR(EIO, "read unallocated block");
      return -1;
    }

  return readlblock(vol, vol->mdb.drAlBlSt + anum * vol->lpa + index, ptr);
}

/*
 * NAME:	writeablock()
 * DESCRIPTION:	write an allocation block to a volume
 */
static
int writeablock(hfsvol *vol,
		unsigned int anum, unsigned int index, block *ptr)
{
  /* verify the allocation block exists and is marked as in-use */

  if (anum >= vol->mdb.drNmAlBlks)
    {
      ERROR(EIO, "write nonexistent block");
      return -1;
    }

  if (! HFS_BMTST(vol->vbm, anum))
    {
      ERROR(EIO, "write unallocated block");
      return -1;
    }

  return writelblock(vol, vol->mdb.drAlBlSt + anum * vol->lpa + index, ptr);
}

/* Low-Level Intelligence Routines ========================================= */

/*
 * NAME:	readpm()
 * DESCRIPTION:	read the partition map and locate an HFS volume
 */
static
int readpm(hfsvol *vol, int pnum)
{
  block b;
  char *ptr;
  Partition map;
  unsigned long bnum;

  bnum = 1;

  while (1)
    {
      if (readlblock(vol, bnum, &b) < 0)
	return -1;

      ptr = (char *) &b;

      fetchw(&ptr, &map.pmSig);
      fetchw(&ptr, &map.pmSigPad);
      fetchl(&ptr, &map.pmMapBlkCnt);
      fetchl(&ptr, &map.pmPyPartStart);
      fetchl(&ptr, &map.pmPartBlkCnt);

      memcpy(map.pmPartName, ptr, 32);
      map.pmPartName[32] = 0;
      ptr += 32;

      memcpy(map.pmParType, ptr, 32);
      map.pmParType[32] = 0;
      ptr += 32;

      fetchl(&ptr, &map.pmLgDataStart);
      fetchl(&ptr, &map.pmDataCnt);
      fetchl(&ptr, &map.pmPartStatus);
      fetchl(&ptr, &map.pmLgBootStart);
      fetchl(&ptr, &map.pmBootSize);
      fetchl(&ptr, &map.pmBootAddr);
      fetchl(&ptr, &map.pmBootAddr2);
      fetchl(&ptr, &map.pmBootEntry);
      fetchl(&ptr, &map.pmBootEntry2);
      fetchl(&ptr, &map.pmBootCksum);

      memcpy(map.pmProcessor, ptr, 16);
      map.pmProcessor[16] = 0;
      ptr += 16;

      if (map.pmSig == 0x5453)
	{
	  /* old partition map sig */

	  ERROR(EINVAL, "unsupported partition map signature");
	  return -1;
	}

      if (map.pmSig != 0x504d)
	{
	  ERROR(EINVAL, "invalid partition map");
	  return -1;
	}

      if (strcmp(map.pmParType, "Apple_HFS") == 0 && --pnum == 0)
	{
	  if (map.pmLgDataStart != 0)
	    {
	      ERROR(EINVAL, "unsupported start of partition logical data");
	      return -1;
	    }

	  vol->vstart = map.pmPyPartStart;
	  vol->vlen   = map.pmPartBlkCnt;

	  return 0;
	}

      if (bnum >= map.pmMapBlkCnt)
	{
	  ERROR(EINVAL, "can't find HFS partition");
	  return -1;
	}

      ++bnum;
    }
}

/*
 * NAME:	readblock0()
 * DESCRIPTION:	read the first sector and get bearings
 */
static
int readblock0(hfsvol *vol, int pnum)
{
  block b;
  char *ptr = (char *) &b;
  Block0 rec;

  if (readlblock(vol, 0, &b) < 0)
    return -1;

  fetchw(&ptr, &rec.sbSig);
  fetchw(&ptr, &rec.sbBlkSize);
  fetchl(&ptr, &rec.sbBlkCount);
  fetchw(&ptr, &rec.sbDevType);
  fetchw(&ptr, &rec.sbDevId);
  fetchl(&ptr, &rec.sbData);
  fetchw(&ptr, &rec.sbDrvrCount);
  fetchl(&ptr, &rec.ddBlock);
  fetchw(&ptr, &rec.ddSize);
  fetchw(&ptr, &rec.ddType);

  switch (rec.sbSig)
    {
    case 0x4552:  /* block device with a partition table */
      {
	if (rec.sbBlkSize != HFS_BLOCKSZ)
	  {
	    ERROR(EINVAL, "unsupported block size");
	    return -1;
	  }

	vol->vlen = rec.sbBlkCount;

	if (readpm(vol, pnum) < 0)
	  return -1;
      }
      break;

    case 0x4c4b:  /* bootable floppy */
    case 0x0000:  /* non-bootable floppy */
      break;

    default:  /* not sure what it is; hope we find a MDB */
      ;
    }

  return 0;
}

/*
 * NAME:	readmdb()
 * DESCRIPTION:	read the master directory block into memory
 */
static
int readmdb(hfsvol *vol)
{
  block b;
  char *ptr = (char *) &b;
  MDB *mdb = &vol->mdb;
  int i;

  if (readlblock(vol, 2, &b) < 0)
    return -1;

  fetchw(&ptr, &mdb->drSigWord);
  fetchl(&ptr, &mdb->drCrDate);
  fetchl(&ptr, &mdb->drLsMod);
  fetchw(&ptr, &mdb->drAtrb);
  fetchw(&ptr, &mdb->drNmFls);
  fetchw(&ptr, &mdb->drVBMSt);
  fetchw(&ptr, &mdb->drAllocPtr);
  fetchw(&ptr, &mdb->drNmAlBlks);
  fetchl(&ptr, &mdb->drAlBlkSiz);
  fetchl(&ptr, &mdb->drClpSiz);
  fetchw(&ptr, &mdb->drAlBlSt);
  fetchl(&ptr, &mdb->drNxtCNID);
  fetchw(&ptr, &mdb->drFreeBks);

  fetchs(&ptr, mdb->drVN, sizeof(mdb->drVN));

  if (ptr - (char *) &b != 64)
    abort();

  fetchl(&ptr, &mdb->drVolBkUp);
  fetchw(&ptr, &mdb->drVSeqNum);
  fetchl(&ptr, &mdb->drWrCnt);
  fetchl(&ptr, &mdb->drXTClpSiz);
  fetchl(&ptr, &mdb->drCTClpSiz);
  fetchw(&ptr, &mdb->drNmRtDirs);
  fetchl(&ptr, &mdb->drFilCnt);
  fetchl(&ptr, &mdb->drDirCnt);

  memcpy(mdb->drFndrInfo, ptr, sizeof(mdb->drFndrInfo));
  ptr += sizeof(mdb->drFndrInfo);

  if (ptr - (char *) &b != 124)
    abort();

  fetchw(&ptr, &mdb->drVCSize);
  fetchw(&ptr, &mdb->drVCBMSize);
  fetchw(&ptr, &mdb->drCtlCSize);
  fetchl(&ptr, &mdb->drXTFlSize);

  for (i = 0; i < 3; ++i)
    {
      fetchw(&ptr, &mdb->drXTExtRec[i].xdrStABN);
      fetchw(&ptr, &mdb->drXTExtRec[i].xdrNumABlks);
    }

  if (ptr - (char *) &b != 146)
    abort();

  fetchl(&ptr, &mdb->drCTFlSize);

  for (i = 0; i < 3; ++i)
    {
      fetchw(&ptr, &mdb->drCTExtRec[i].xdrStABN);
      fetchw(&ptr, &mdb->drCTExtRec[i].xdrNumABlks);
    }

  if (ptr - (char *) &b != 162)
    abort();

  if (mdb->drAlBlkSiz % HFS_BLOCKSZ != 0)
    {
      ERROR(EINVAL, "bad allocation block size");
      return -1;
    }

  vol->lpa = mdb->drAlBlkSiz / HFS_BLOCKSZ;

  return 0;
}

/*
 * NAME:	writemdb()
 * DESCRIPTION:	write the master directory block to disk
 */
static
int writemdb(hfsvol *vol)
{
  block b;
  char *ptr = (char *) &b;
  MDB *mdb = &vol->mdb;
  int i;

  storew(&ptr, mdb->drSigWord);
  storel(&ptr, mdb->drCrDate);
  storel(&ptr, mdb->drLsMod);
  storew(&ptr, mdb->drAtrb);
  storew(&ptr, mdb->drNmFls);
  storew(&ptr, mdb->drVBMSt);
  storew(&ptr, mdb->drAllocPtr);
  storew(&ptr, mdb->drNmAlBlks);
  storel(&ptr, mdb->drAlBlkSiz);
  storel(&ptr, mdb->drClpSiz);
  storew(&ptr, mdb->drAlBlSt);
  storel(&ptr, mdb->drNxtCNID);
  storew(&ptr, mdb->drFreeBks);

  storeb(&ptr, strlen(mdb->drVN));
  memcpy(ptr, mdb->drVN, sizeof(mdb->drVN) - 1);

  ptr += sizeof(mdb->drVN) - 1;

  if (ptr - (char *) &b != 64)
    abort();

  storel(&ptr, mdb->drVolBkUp);
  storew(&ptr, mdb->drVSeqNum);
  storel(&ptr, mdb->drWrCnt);
  storel(&ptr, mdb->drXTClpSiz);
  storel(&ptr, mdb->drCTClpSiz);
  storew(&ptr, mdb->drNmRtDirs);
  storel(&ptr, mdb->drFilCnt);
  storel(&ptr, mdb->drDirCnt);

  memcpy(ptr, mdb->drFndrInfo, sizeof(mdb->drFndrInfo));
  ptr += sizeof(mdb->drFndrInfo);

  if (ptr - (char *) &b != 124)
    abort();

  storew(&ptr, mdb->drVCSize);
  storew(&ptr, mdb->drVCBMSize);
  storew(&ptr, mdb->drCtlCSize);
  storel(&ptr, mdb->drXTFlSize);

  for (i = 0; i < 3; ++i)
    {
      storew(&ptr, mdb->drXTExtRec[i].xdrStABN);
      storew(&ptr, mdb->drXTExtRec[i].xdrNumABlks);
    }

  if (ptr - (char *) &b != 146)
    abort();

  storel(&ptr, mdb->drCTFlSize);

  for (i = 0; i < 3; ++i)
    {
      storew(&ptr, mdb->drCTExtRec[i].xdrStABN);
      storew(&ptr, mdb->drCTExtRec[i].xdrNumABlks);
    }

  if (ptr - (char *) &b != 162)
    abort();

  if (vol->flags & HFS_UPDATE_ALTMDB)
    {
      if (writelblock(vol, vol->vlen - 2, &b) < 0)
	return -1;
    }

  if (writelblock(vol, 2, &b) < 0)
    return -1;

  vol->flags &= ~(HFS_UPDATE_MDB | HFS_UPDATE_ALTMDB);

  return 0;
}

/*
 * NAME:	readvbm()
 * DESCRIPTION:	read the volume bit map into memory
 */
static
int readvbm(hfsvol *vol)
{
  int vbmst = vol->mdb.drVBMSt;
  int vbmsz = (vol->mdb.drNmAlBlks + 4095) / 4096;
  block *b;

  if (vol->mdb.drAlBlSt - vbmst < vbmsz)
    {
      ERROR(EIO, "volume bitmap collides with volume data");
      return -1;
    }

  b = ALLOC(block, vbmsz);
  if (b == 0)
    {
      ERROR(ENOMEM, "out of memory");
      return -1;
    }

  vol->vbm = b;

  while (vbmsz--)
    if (readlblock(vol, vbmst++, b++) < 0)
      {
	FREE(vol->vbm);
	return -1;
      }

  return 0;
}

/*
 * NAME:	writevbm()
 * DESCRIPTION:	write the volume bit map to disk
 */
static
int writevbm(hfsvol *vol)
{
  int vbmst = vol->mdb.drVBMSt;
  int vbmsz = (vol->mdb.drNmAlBlks + 4095) / 4096;
  block *b = vol->vbm;

  while (vbmsz--)
    {
      if (writelblock(vol, vbmst++, b++) < 0)
	return -1;
    }

  vol->flags &= ~HFS_UPDATE_VBM;

  return 0;
}

/*
 * NAME:	dofblock()
 * DESCRIPTION:	read or write a numbered block from a file
 */
static
int dofblock(hfsfile *file, unsigned long num, block *ptr,
	     int (*func)(hfsvol *, unsigned int, unsigned int, block *))
{
  unsigned int abnum;
  unsigned int blnum;
  ExtDescriptor *exts;
  unsigned int nexts;
  unsigned int i;

  abnum = num / file->vol->lpa;
  blnum = num % file->vol->lpa;

  switch (file->fork)
    {
    case fkRsrc:
      exts  = file->rexts;
      nexts = file->nrexts;
      break;

    case fkData:
    default:
      exts  = file->dexts;
      nexts = file->ndexts;
    }

  /* locate the appropriate extent record */

  for (i = 0; i < nexts; ++i)
    {
      if (abnum < exts[i].xdrNumABlks)
	return (*func)(file->vol, exts[i].xdrStABN + abnum, blnum, ptr);

      abnum -= exts[i].xdrNumABlks;
    }

  ERROR(EIO, "extent not found");
  return -1;
}

# define getfblock(file, num, ptr)	dofblock(file, num, ptr, readablock)
# define putfblock(file, num, ptr)	dofblock(file, num, ptr, writeablock)

/* B*-tree routines ======================================================== */

/*
 * NAME:	getnode()
 * DESCRIPTION:	retrieve a numbered node from a B*-tree file
 */
static
int getnode(node *n)
{
  btree *bt = n->bt;
  block *bp = &n->data;
  char *ptr;
  int i;

  /* verify the node exists and is marked as in-use */

  if (n->nnum && n->nnum >= bt->hdr.bthNNodes)
    {
      ERROR(EIO, "read nonexistent b*-tree node");
      return -1;
    }

  if (bt->map && ! HFS_BMTST(bt->map, n->nnum))
    {
      ERROR(EIO, "read unallocated b*-tree node");
      return -1;
    }

  if (getfblock(&bt->f, n->nnum, bp) < 0)
    return -1;

  ptr = (char *) bp;

  fetchl(&ptr, &n->nd.ndFLink);
  fetchl(&ptr, &n->nd.ndBLink);
  fetchb(&ptr, &n->nd.ndType);
  fetchb(&ptr, &n->nd.ndNHeight);
  fetchw(&ptr, &n->nd.ndNRecs);
  fetchw(&ptr, &n->nd.ndResv2);

  i = n->nd.ndNRecs + 1;

  if (i > sizeof(n->roff) / sizeof(n->roff[0]))
    {
      ERROR(EIO, "bad b*-tree node descriptor");
      return -1;
    }

  ptr = (char *) bp + HFS_BLOCKSZ - SIZE(word, i);

  while (i--)
    fetchw(&ptr, &n->roff[i]);

  return 0;
}

/*
 * NAME:	putnode()
 * DESCRIPTION:	store a numbered node into a B*-tree file
 */
static
int putnode(node *n)
{
  btree *bt = n->bt;
  block *bp = &n->data;
  char *ptr;
  int i;

  /* verify the node exists and is marked as in-use */

  if (n->nnum && n->nnum >= bt->hdr.bthNNodes)
    {
      ERROR(EIO, "write nonexistent b*-tree node");
      return -1;
    }

  if (bt->map && ! HFS_BMTST(bt->map, n->nnum))
    {
      ERROR(EIO, "write unallocated b*-tree node");
      return -1;
    }

  ptr = (char *) bp;

  storel(&ptr, n->nd.ndFLink);
  storel(&ptr, n->nd.ndBLink);
  storeb(&ptr, n->nd.ndType);
  storeb(&ptr, n->nd.ndNHeight);
  storew(&ptr, n->nd.ndNRecs);
  storew(&ptr, n->nd.ndResv2);

  i = n->nd.ndNRecs + 1;

  if (i > sizeof(n->roff) / sizeof(n->roff[0]))
    {
      ERROR(EIO, "bad b*-tree node descriptor");
      return -1;
    }

  ptr = (char *) bp + HFS_BLOCKSZ - SIZE(word, i);

  while (i--)
    storew(&ptr, n->roff[i]);

  if (putfblock(&bt->f, n->nnum, bp) < 0)
    return -1;

  return 0;
}

static int getextents(hfsfile *);

/*
 * NAME:	readbthdr()
 * DESCRIPTION:	read the header node of a B*-tree
 */
static
int readbthdr(btree *bt)
{
  char *ptr, *map;
  int i;
  unsigned long nnum;

  bt->hdrnd.bt   = bt;
  bt->hdrnd.nnum = 0;

  if (getnode(&bt->hdrnd) < 0)
    return -1;

  if (bt->hdrnd.nd.ndType != ndHdrNode ||
      bt->hdrnd.nd.ndNRecs != 3 ||
      bt->hdrnd.roff[0] != 0x00e ||
      bt->hdrnd.roff[1] != 0x078 ||
      bt->hdrnd.roff[2] != 0x0f8 ||
      bt->hdrnd.roff[3] != 0x1f8)
    {
      ERROR(EIO, "malformed b*-tree header node");
      return -1;
    }

  /* read header record */

  ptr = HFS_NODEREC(bt->hdrnd, 0);

  fetchw(&ptr, &bt->hdr.bthDepth);
  fetchl(&ptr, &bt->hdr.bthRoot);
  fetchl(&ptr, &bt->hdr.bthNRecs);
  fetchl(&ptr, &bt->hdr.bthFNode);
  fetchl(&ptr, &bt->hdr.bthLNode);
  fetchw(&ptr, &bt->hdr.bthNodeSize);
  fetchw(&ptr, &bt->hdr.bthKeyLen);
  fetchl(&ptr, &bt->hdr.bthNNodes);
  fetchl(&ptr, &bt->hdr.bthFree);

  for (i = 0; i < 76; ++i)
    fetchb(&ptr, &bt->hdr.bthResv[i]);

  if (bt->hdr.bthNodeSize != HFS_BLOCKSZ)
    {
      ERROR(EINVAL, "unsupported b*-tree node size");
      return -1;
    }

  /* construct btree file extents */

  if (getextents(&bt->f) < 0)
    return -1;

  /* read map record; construct btree bitmap */
  /* don't set bt->map until we're done, since getnode() checks it */

  ptr = HFS_NODEREC(bt->hdrnd, 2);

  map = ALLOC(char, HFS_MAP1SZ);
  if (map == 0)
    {
      ERROR(ENOMEM, "out of memory");
      return -1;
    }

  memcpy(map, ptr, HFS_MAP1SZ);
  bt->mapsz = HFS_MAP1SZ;

  /* read continuation map records, if any */

  nnum = bt->hdrnd.nd.ndFLink;

  while (nnum)
    {
      node n;
      char *newmap;

      n.bt   = bt;
      n.nnum = nnum;

      if (getnode(&n) < 0)
	return -1;

      if (n.nd.ndType != ndMapNode ||
	  n.nd.ndNRecs != 1 ||
	  n.roff[0] != 0x00e ||
	  n.roff[1] != 0x1fa)
	{
	  FREE(map);
	  ERROR(EIO, "malformed b*-tree map node");
	  return -1;
	}

      newmap = REALLOC(map, char, bt->mapsz + HFS_MAPXSZ);
      if (newmap == 0)
	{
	  FREE(map);
	  ERROR(ENOMEM, "out of memory");
	  return -1;
	}

      memcpy(newmap + bt->mapsz, HFS_NODEREC(n, 0), HFS_MAPXSZ);

      map = newmap;
      bt->mapsz += HFS_MAPXSZ;

      nnum = n.nd.ndFLink;
    }

  bt->map = map;

  return 0;
}

/*
 * NAME:	writebthdr()
 * DESCRIPTION:	write the header node of a B*-tree
 */
static
int writebthdr(btree *bt)
{
  char *ptr, *map;
  unsigned long mapsz, nnum;
  int i;

  if (bt->hdrnd.nnum != 0 ||
      bt->hdrnd.nd.ndType != ndHdrNode)
    abort();

  bt->hdrnd.nd.ndNRecs = 3;

  ptr = HFS_NODEREC(bt->hdrnd, 0);

  storew(&ptr, bt->hdr.bthDepth);
  storel(&ptr, bt->hdr.bthRoot);
  storel(&ptr, bt->hdr.bthNRecs);
  storel(&ptr, bt->hdr.bthFNode);
  storel(&ptr, bt->hdr.bthLNode);
  storew(&ptr, bt->hdr.bthNodeSize);
  storew(&ptr, bt->hdr.bthKeyLen);
  storel(&ptr, bt->hdr.bthNNodes);
  storel(&ptr, bt->hdr.bthFree);

  for (i = 0; i < 76; ++i)
    storeb(&ptr, bt->hdr.bthResv[i]);

  memcpy(HFS_NODEREC(bt->hdrnd, 2), bt->map, HFS_MAP1SZ);

  bt->hdrnd.bt = bt;

  if (putnode(&bt->hdrnd) < 0)
    return -1;

  map   = bt->map   + HFS_MAP1SZ;
  mapsz = bt->mapsz - HFS_MAP1SZ;

  nnum  = bt->hdrnd.nd.ndFLink;

  while (mapsz)
    {
      node n;

      if (nnum == 0)
	{
	  ERROR(EIO, "truncated b*-tree map");
	  return -1;
	}

      n.bt   = bt;
      n.nnum = nnum;

      if (getnode(&n) < 0)
	return -1;

      if (n.nd.ndType != ndMapNode ||
	  n.nd.ndNRecs != 1 ||
	  n.roff[0] != 0x00e ||
	  n.roff[1] != 0x1fa)
	{
	  ERROR(EIO, "malformed b*-tree map node");
	  return -1;
	}

      memcpy(HFS_NODEREC(n, 0), map, HFS_MAPXSZ);

      if (putnode(&n) < 0)
	return -1;

      map   += HFS_MAPXSZ;
      mapsz -= HFS_MAPXSZ;

      nnum = n.nd.ndFLink;
    }

  bt->flags &= ~HFS_UPDATE_BTHDR;

  return 0;
}

/*
 * NAME:	newnode()
 * DESCRIPTION:	allocate a new b*-tree node
 */
static
int newnode(node *np)
{
  btree *bt = np->bt;
  unsigned long num;

  if (bt->hdr.bthFree == 0)
    {
      /* must extend tree file */

      ERROR(ENOSYS, "b*-tree extension not yet implemented");
      return -1;
    }

  num = 0;
  while (num < bt->hdr.bthNNodes && HFS_BMTST(bt->map, num))
    ++num;

  if (num == bt->hdr.bthNNodes)
    {
      ERROR(EIO, "free b*-tree node not found");
      return -1;
    }

  HFS_BMSET(bt->map, num);
  --bt->hdr.bthFree;

  bt->flags |= HFS_UPDATE_BTHDR;

  np->nnum = num;

  np->nd.ndNRecs = 0;
  np->nd.ndResv2 = 0;

  np->roff[0] = 0x00e;

  return 0;
}

/*
 * NAME:	compactnode()
 * DESCRIPTION:	remove deleted records from a node
 */
static
void compactnode(node *np)
{
  char *ptr;
  int offset, nrecs, i;

  offset = 0x00e;
  ptr    = (char *) &np->data + offset;
  nrecs  = 0;

  for (i = 0; i < np->nd.ndNRecs; ++i)
    {
      char *rec;
      int reclen;

      rec    = HFS_NODEREC(*np, i);
      reclen = np->roff[i + 1] - np->roff[i];

      if (HFS_RECKEYLEN(rec) > 0)
	{
	  np->roff[nrecs++] = offset;
	  offset += reclen;

	  if (ptr == rec)
	    ptr += reclen;
	  else
	    {
	      while (reclen--)
		*ptr++ = *rec++;
	    }
	}
    }

  np->roff[nrecs] = offset;
  np->nd.ndNRecs  = nrecs;
}

/*
 * NAME:	treesearch()
 * DESCRIPTION:	locate a data record given a search key
 */
static
int treesearch(node *np, char *searchkey)
{
  btree *bt = np->bt;
  int i;

  np->nnum = bt->hdr.bthRoot;

  if (np->nnum == 0)
    {
      ERROR(ENOENT, "empty b*-tree");
      return -1;
    }

  while (1)
    {
      int comp = -1;
      char *rec = 0;

      if (getnode(np) < 0)
	return -1;

      for (i = np->nd.ndNRecs; i--; )
	{
	  rec = HFS_NODEREC(*np, i);

	  if (HFS_RECKEYLEN(rec) == 0)
	    continue;  /* deleted record */

	  comp = bt->compare(bt->unpack(rec), searchkey);

	  if (comp <= 0)
	    break;
	}

      switch (np->nd.ndType)
	{
	case ndIndxNode:
	  if (rec == 0)
	    {
	      ERROR(EIO, "bad b*-tree structure");
	      return -1;
	    }
	  else
	    np->nnum = getml(HFS_RECDATA(rec));

	  break;

	case ndLeafNode:
	  np->rnum = i;

	  if (comp == 0)
	    return 0;
	  else
	    {
	      ERROR(ENOENT, "no such file or directory");
	      return -1;
	    }

	  break;

	default:
	  ERROR(EIO, "bad b*-tree structure");
	  return -1;
	}
    }
}

/*
 * NAME:	treeinsert()
 * DESCRIPTION:	insert a new node record into a tree
 */
static
int treeinsert(btree *bt, char *record, int reclen)
{
  node n;
  int rnum, i;
  char *ptr;
  CatKeyRec key;

  n.bt   = bt;
  n.nnum = bt->hdr.bthRoot;

  if (n.nnum == 0)
    {
      /* create root node */

      n.nd.ndFLink   = 0;
      n.nd.ndBLink   = 0;
      n.nd.ndType    = ndLeafNode;
      n.nd.ndNHeight = 1;

      if (newnode(&n) < 0)
	return -1;

      if (putnode(&n) < 0)
	return -1;

      bt->hdr.bthDepth = 1;
      bt->hdr.bthRoot  = n.nnum;
      bt->hdr.bthFNode = n.nnum;
      bt->hdr.bthLNode = n.nnum;

      bt->flags |= HFS_UPDATE_BTHDR;
    }

  /* find leaf node for insertion */

  memcpy((char *) &key, bt->unpack(record), sizeof(key));
  if (treesearch(&n, (char *) &key) >= 0)
    {
      ERROR(EIO, "b*-tree record already exists");
      return -1;
    }
  else if (errno != ENOENT)
    return -1;

  /* see if node has space for the new record */

  compactnode(&n);

  if (reclen + sizeof(word) >
      HFS_BLOCKSZ - n.roff[n.nd.ndNRecs] - SIZE(word, n.nd.ndNRecs + 1))
    {
      /* node must be split */

      ERROR(ENOSYS, "splitting b*-tree nodes not yet implemented");
      return -1;
    }

  /* insert the record */

  rnum = n.rnum + 1;

  for (ptr = HFS_NODEREC(n, n.nd.ndNRecs) + reclen;
       ptr > HFS_NODEREC(n, rnum) + reclen; --ptr)
    *(ptr - 1) = *(ptr - 1 - reclen);

  ++n.nd.ndNRecs;

  for (i = n.nd.ndNRecs; i > rnum; --i)
    n.roff[i] = n.roff[i - 1] + reclen;

  memcpy(HFS_NODEREC(n, rnum), record, reclen);

  /* store the modified node */

  if (putnode(&n) < 0)
    return -1;

  ++bt->hdr.bthNRecs;

  bt->flags |= HFS_UPDATE_BTHDR;

  return 0;
}

/*
 * NAME:	catkeycompare()
 * DESCRIPTION:	compare two catalog record keys
 */
static
int catkeycompare(char *vkey1, char *vkey2)
{
  char *ptr1, *ptr2;
  CatKeyRec *key1 = (CatKeyRec *) vkey1;
  CatKeyRec *key2 = (CatKeyRec *) vkey2;

  if (key1->ckrParID < key2->ckrParID)
    return -1;
  else if (key1->ckrParID > key2->ckrParID)
    return 1;

  /* this is a specialized strcasecmp() */

  for (ptr1 = key1->ckrCName, ptr2 = key2->ckrCName;
       *ptr1 && *ptr2; ++ptr1, ++ptr2)
    {
      int c1, c2;

      c1 = toupper(*ptr1);
      c2 = toupper(*ptr2);

      if (c1 < c2)
	return -1;
      else if (c1 > c2)
	return 1;
    }

  if (! *ptr1 && *ptr2)
    return -1;
  else if (*ptr1 && ! *ptr2)
    return 1;

  return 0;
}

/*
 * NAME:	extkeycompare()
 * DESCRIPTION:	compare two extents record keys
 */
static
int extkeycompare(char *vkey1, char *vkey2)
{
  ExtKeyRec *key1 = (ExtKeyRec *) vkey1;
  ExtKeyRec *key2 = (ExtKeyRec *) vkey2;

  if (key1->xkrFkType < key2->xkrFkType)
    return -1;
  else if (key1->xkrFkType > key2->xkrFkType)
    return 1;

  if (key1->xkrFNum < key2->xkrFNum)
    return -1;
  else if (key1->xkrFNum > key2->xkrFNum)
    return 1;

  if (key1->xkrFABN < key2->xkrFABN)
    return -1;
  else if (key1->xkrFABN > key2->xkrFABN)
    return 1;

  return 0;
}

/*
 * NAME:	catkeyunpack()
 * DESCRIPTION:	unpack a catalog record key
 */
static
char *catkeyunpack(char *pkey)
{
  static CatKeyRec key;

  fetchb(&pkey, &key.ckrKeyLen);
  fetchb(&pkey, &key.ckrResrv1);
  fetchl(&pkey, &key.ckrParID);
  fetchs(&pkey, key.ckrCName, sizeof(key.ckrCName));

  return (char *) &key;
}

/*
 * NAME:	extkeyunpack()
 * DESCRIPTION:	unpack an extents record key
 */
static
char *extkeyunpack(char *pkey)
{
  static ExtKeyRec key;

  fetchb(&pkey, &key.xkrKeyLen);
  fetchb(&pkey, &key.xkrFkType);
  fetchl(&pkey, &key.xkrFNum);
  fetchw(&pkey, &key.xkrFABN);

  return (char *) &key;
}

/*
 * NAME:	catdataunpack()
 * DESCRIPTION:	unpack catalog record data
 */
static
void catdataunpack(char *ptr, CatDataRec *rec)
{
  int i;

  fetchb(&ptr, &rec->cdrType);
  fetchb(&ptr, &rec->cdrResrv2);

  switch (rec->cdrType)
    {
    case cdrDirRec:
      fetchw(&ptr, &rec->u.dir.dirFlags);
      fetchw(&ptr, &rec->u.dir.dirVal);
      fetchl(&ptr, &rec->u.dir.dirDirID);
      fetchl(&ptr, &rec->u.dir.dirCrDat);
      fetchl(&ptr, &rec->u.dir.dirMdDat);
      fetchl(&ptr, &rec->u.dir.dirBkDat);

      fetchw(&ptr, &rec->u.dir.dirUsrInfo.frRect.top);
      fetchw(&ptr, &rec->u.dir.dirUsrInfo.frRect.left);
      fetchw(&ptr, &rec->u.dir.dirUsrInfo.frRect.bottom);
      fetchw(&ptr, &rec->u.dir.dirUsrInfo.frRect.right);
      fetchw(&ptr, &rec->u.dir.dirUsrInfo.frFlags);
      fetchw(&ptr, &rec->u.dir.dirUsrInfo.frLocation.v);
      fetchw(&ptr, &rec->u.dir.dirUsrInfo.frLocation.h);
      fetchw(&ptr, &rec->u.dir.dirUsrInfo.frView);

      fetchw(&ptr, &rec->u.dir.dirFndrInfo.frScroll.v);
      fetchw(&ptr, &rec->u.dir.dirFndrInfo.frScroll.h);
      fetchl(&ptr, &rec->u.dir.dirFndrInfo.frOpenChain);
      fetchw(&ptr, &rec->u.dir.dirFndrInfo.frUnused);
      fetchw(&ptr, &rec->u.dir.dirFndrInfo.frComment);
      fetchl(&ptr, &rec->u.dir.dirFndrInfo.frPutAway);

      for (i = 0; i < 4; ++i)
	fetchl(&ptr, &rec->u.dir.dirResrv[i]);

      break;

    case cdrFilRec:
      fetchb(&ptr, &rec->u.fil.filFlags);
      fetchb(&ptr, &rec->u.fil.filTyp);

      fetchl(&ptr, &rec->u.fil.filUsrWds.fdType);
      fetchl(&ptr, &rec->u.fil.filUsrWds.fdCreator);
      fetchw(&ptr, &rec->u.fil.filUsrWds.fdFlags);
      fetchw(&ptr, &rec->u.fil.filUsrWds.fdLocation.v);
      fetchw(&ptr, &rec->u.fil.filUsrWds.fdLocation.h);
      fetchw(&ptr, &rec->u.fil.filUsrWds.fdFldr);

      fetchl(&ptr, &rec->u.fil.filFlNum);

      fetchw(&ptr, &rec->u.fil.filStBlk);
      fetchl(&ptr, &rec->u.fil.filLgLen);
      fetchl(&ptr, &rec->u.fil.filPyLen);

      fetchw(&ptr, &rec->u.fil.filRStBlk);
      fetchl(&ptr, &rec->u.fil.filRLgLen);
      fetchl(&ptr, &rec->u.fil.filRPyLen);

      fetchl(&ptr, &rec->u.fil.filCrDat);
      fetchl(&ptr, &rec->u.fil.filMdDat);
      fetchl(&ptr, &rec->u.fil.filBkDat);

      fetchw(&ptr, &rec->u.fil.filFndrInfo.fdIconID);
      for (i = 0; i < 4; ++i)
	fetchw(&ptr, &rec->u.fil.filFndrInfo.fdUnused[i]);
      fetchw(&ptr, &rec->u.fil.filFndrInfo.fdComment);
      fetchl(&ptr, &rec->u.fil.filFndrInfo.fdPutAway);

      fetchw(&ptr, &rec->u.fil.filClpSize);

      for (i = 0; i < 3; ++i)
	{
	  fetchw(&ptr, &rec->u.fil.filExtRec[i].xdrStABN);
	  fetchw(&ptr, &rec->u.fil.filExtRec[i].xdrNumABlks);
	}

      for (i = 0; i < 3; ++i)
	{
	  fetchw(&ptr, &rec->u.fil.filRExtRec[i].xdrStABN);
	  fetchw(&ptr, &rec->u.fil.filRExtRec[i].xdrNumABlks);
	}

      fetchl(&ptr, &rec->u.fil.filResrv);

      break;

    case cdrThdRec:
      for (i = 0; i < 2; ++i)
	fetchl(&ptr, &rec->u.dthd.thdResrv[i]);

      fetchl(&ptr, &rec->u.dthd.thdParID);
      fetchs(&ptr, rec->u.dthd.thdCName, sizeof(rec->u.dthd.thdCName));

      break;

    case cdrFThdRec:
      for (i = 0; i < 2; ++i)
	fetchl(&ptr, &rec->u.fthd.fthdResrv[i]);

      fetchl(&ptr, &rec->u.fthd.fthdParID);
      fetchs(&ptr, rec->u.fthd.fthdCName, sizeof(rec->u.fthd.fthdCName));

      break;
    }
}

/*
 * NAME:	extdataunpack()
 * DESCRIPTION:	unpack extents record data
 */
static
void extdataunpack(char *ptr, ExtDataRec *rec)
{
  int i;

  for (i = 0; i < 3; ++i)
    {
      fetchw(&ptr, &(*rec)[i].xdrStABN);
      fetchw(&ptr, &(*rec)[i].xdrNumABlks);
    }
}

/*
 * NAME:	makecatkey()
 * DESCRIPTION:	construct a catalog record key
 */
static
void makecatkey(CatKeyRec *key, long parid, char *name, int pad)
{
  if (pad)
    key->ckrKeyLen = 0x25;
  else
    {
      int len;

      len = strlen(name);
      key->ckrKeyLen = 0x07 + len + (len & 1);
    }

  key->ckrResrv1 = 0;
  key->ckrParID  = parid;

  strcpy(key->ckrCName, name);
}

/*
 * NAME:	makeextkey()
 * DESCRIPTION:	construct an extents record key
 */
static
void makeextkey(ExtKeyRec *key, int fork, long fnum, int fabn)
{
  key->xkrKeyLen = 0x07;
  key->xkrFkType = fork;
  key->xkrFNum   = fnum;
  key->xkrFABN   = fabn;
}

/*
 * NAME:	makecatrec()
 * DESCRIPTION:	construct a complete catalog record
 */
static
char *makecatrec(CatKeyRec *key, CatDataRec *data, int *len)
{
  static char rec[sizeof(CatKeyRec) + 1 + sizeof(CatDataRec)];
  char *ptr = rec;
  int i;

  storeb(&ptr, key->ckrKeyLen);
  storeb(&ptr, key->ckrResrv1);
  storel(&ptr, key->ckrParID);

  memset(ptr, 0, sizeof(key->ckrCName));
  stores(&ptr, key->ckrCName, sizeof(key->ckrCName));
  storeb(&ptr, 0);  /* key pad */

  /* realign for record data; key may or may not be padded */

  ptr = HFS_RECDATA(rec);

  storeb(&ptr, data->cdrType);
  storeb(&ptr, data->cdrResrv2);

  switch (data->cdrType)
    {
    case cdrDirRec:
      storew(&ptr, data->u.dir.dirFlags);
      storew(&ptr, data->u.dir.dirVal);
      storel(&ptr, data->u.dir.dirDirID);
      storel(&ptr, data->u.dir.dirCrDat);
      storel(&ptr, data->u.dir.dirMdDat);
      storel(&ptr, data->u.dir.dirBkDat);

      storew(&ptr, data->u.dir.dirUsrInfo.frRect.top);
      storew(&ptr, data->u.dir.dirUsrInfo.frRect.left);
      storew(&ptr, data->u.dir.dirUsrInfo.frRect.bottom);
      storew(&ptr, data->u.dir.dirUsrInfo.frRect.right);
      storew(&ptr, data->u.dir.dirUsrInfo.frFlags);
      storew(&ptr, data->u.dir.dirUsrInfo.frLocation.v);
      storew(&ptr, data->u.dir.dirUsrInfo.frLocation.h);
      storew(&ptr, data->u.dir.dirUsrInfo.frView);

      storew(&ptr, data->u.dir.dirFndrInfo.frScroll.v);
      storew(&ptr, data->u.dir.dirFndrInfo.frScroll.h);
      storel(&ptr, data->u.dir.dirFndrInfo.frOpenChain);
      storew(&ptr, data->u.dir.dirFndrInfo.frUnused);
      storew(&ptr, data->u.dir.dirFndrInfo.frComment);
      storel(&ptr, data->u.dir.dirFndrInfo.frPutAway);

      for (i = 0; i < 4; ++i)
	storel(&ptr, data->u.dir.dirResrv[i]);

      break;

    case cdrFilRec:
      storeb(&ptr, data->u.fil.filFlags);
      storeb(&ptr, data->u.fil.filTyp);

      storel(&ptr, data->u.fil.filUsrWds.fdType);
      storel(&ptr, data->u.fil.filUsrWds.fdCreator);
      storew(&ptr, data->u.fil.filUsrWds.fdFlags);
      storew(&ptr, data->u.fil.filUsrWds.fdLocation.v);
      storew(&ptr, data->u.fil.filUsrWds.fdLocation.h);
      storew(&ptr, data->u.fil.filUsrWds.fdFldr);

      storel(&ptr, data->u.fil.filFlNum);

      storew(&ptr, data->u.fil.filStBlk);
      storel(&ptr, data->u.fil.filLgLen);
      storel(&ptr, data->u.fil.filPyLen);

      storew(&ptr, data->u.fil.filRStBlk);
      storel(&ptr, data->u.fil.filRLgLen);
      storel(&ptr, data->u.fil.filRPyLen);

      storel(&ptr, data->u.fil.filCrDat);
      storel(&ptr, data->u.fil.filMdDat);
      storel(&ptr, data->u.fil.filBkDat);

      storew(&ptr, data->u.fil.filFndrInfo.fdIconID);
      for (i = 0; i < 4; ++i)
	storew(&ptr, data->u.fil.filFndrInfo.fdUnused[i]);
      storew(&ptr, data->u.fil.filFndrInfo.fdComment);
      storel(&ptr, data->u.fil.filFndrInfo.fdPutAway);

      storew(&ptr, data->u.fil.filClpSize);

      for (i = 0; i < 3; ++i)
	{
	  storew(&ptr, data->u.fil.filExtRec[i].xdrStABN);
	  storew(&ptr, data->u.fil.filExtRec[i].xdrNumABlks);
	}

      for (i = 0; i < 3; ++i)
	{
	  storew(&ptr, data->u.fil.filRExtRec[i].xdrStABN);
	  storew(&ptr, data->u.fil.filRExtRec[i].xdrNumABlks);
	}

      storel(&ptr, data->u.fil.filResrv);

      break;

    case cdrThdRec:
      for (i = 0; i < 2; ++i)
	storel(&ptr, data->u.dthd.thdResrv[i]);

      storel(&ptr, data->u.dthd.thdParID);
      stores(&ptr, data->u.dthd.thdCName, sizeof(data->u.dthd.thdCName));

      break;

    case cdrFThdRec:
      for (i = 0; i < 2; ++i)
	storel(&ptr, data->u.fthd.fthdResrv[i]);

      storel(&ptr, data->u.fthd.fthdParID);
      stores(&ptr, data->u.fthd.fthdCName, sizeof(data->u.fthd.fthdCName));

      break;
    }

  *len = ptr - rec;

  return rec;
}

/*
 * NAME:	makeextrec()
 * DESCRIPTION:	construct a complete extent record
 */
static
char *makeextrec(ExtKeyRec *key, ExtDataRec *data, int *len)
{
  static char rec[sizeof(ExtKeyRec) + 1 + sizeof(ExtDataRec)];
  char *ptr = rec;
  int i;

  storeb(&ptr, key->xkrKeyLen);
  storeb(&ptr, key->xkrFkType);
  storel(&ptr, key->xkrFNum);
  storew(&ptr, key->xkrFABN);
  storeb(&ptr, 0);  /* key pad */

  /* realign for record data; key may or may not be padded */
  /* (unnecessary... extent node record keys are always the same length) */

  ptr = HFS_RECDATA(rec);

  for (i = 0; i < 3; ++i)
    {
      storew(&ptr, (*data)[i].xdrStABN);
      storew(&ptr, (*data)[i].xdrNumABlks);
    }

  *len = ptr - rec;

  return rec;
}

/*
 * NAME:	catsearch()
 * DESCRIPTION:	search catalog tree
 */
static
int catsearch(hfsvol *vol, long parid, char *name,
	      CatDataRec *rec, char *cname, node *np)
{
  CatKeyRec key;
  node n;
  char *ptr;

  if (np == 0)
    np = &n;

  np->bt = &vol->cat;

  makecatkey(&key, parid, name, 0);
  if (treesearch(np, (char *) &key) < 0)
    return -1;

  ptr = HFS_NODEREC(*np, np->rnum);

  if (cname)
    strcpy(cname, ((CatKeyRec *) catkeyunpack(ptr))->ckrCName);

  catdataunpack(HFS_RECDATA(ptr), rec);

  return 0;
}

/*
 * NAME:	extsearch()
 * DESCRIPTION:	search extents tree
 */
static
int extsearch(hfsvol *vol, int fork, long fnum, int fabn,
	      ExtDataRec *rec, node *np)
{
  ExtKeyRec key;
  node n;
  char *ptr;

  if (np == 0)
    np = &n;

  np->bt = &vol->ext;

  makeextkey(&key, fork, fnum, fabn);
  if (treesearch(np, (char *) &key) < 0)
    return -1;

  ptr = HFS_NODEREC(*np, np->rnum);

  extdataunpack(HFS_RECDATA(ptr), rec);

  return 0;
}

/* High-Level Utility Routines ============================================= */

/*
 * NAME:	getdthread()
 * DESCRIPTION:	retrieve catalog thread information for a directory
 */
static
int getdthread(hfsvol *vol, long dirid, CatDataRec *thread)
{
  if (catsearch(vol, dirid, "", thread, 0, 0) < 0)
    return -1;

  if (thread->cdrType != cdrThdRec)
    {
      ERROR(EIO, "thread record isn't");
      return -1;
    }

  return 0;
}

/*
 * NAME:	putcatrec()
 * DESCRIPTION:	store catalog information
 */
static
int putcatrec(CatDataRec *rec, node *np)
{
  char *record, *packed;
  int len;

  record = HFS_NODEREC(*np, np->rnum);
  packed = makecatrec((CatKeyRec *) catkeyunpack(record), rec, &len);

  if (len != np->roff[np->rnum + 1] - np->roff[np->rnum])
    abort();

  memcpy(record, packed, len);

  if (putnode(np) < 0)
    return -1;

  return 0;
}

/*
 * NAME:	putextrec()
 * DESCRIPTION:	store extent information
 */
static
int putextrec(ExtDataRec *rec, node *np)
{
  char *record, *packed;
  int len;

  record = HFS_NODEREC(*np, np->rnum);
  packed = makeextrec((ExtKeyRec *) extkeyunpack(record), rec, &len);

  if (len != np->roff[np->rnum + 1] - np->roff[np->rnum])
    abort();

  memcpy(record, packed, len);

  if (putnode(np) < 0)
    return -1;

  return 0;
}

/*
 * NAME:	getextents()
 * DESCRIPTION:	find all extents for a file
 */
static
int getextents(hfsfile *file)
{
  unsigned int nexts, totablks, ablkcnt;
  ExtDataRec rec, *recp;
  int chunkcnt;

  file->dexts  = 0;
  file->ndexts = 0;

  file->rexts  = 0;
  file->nrexts = 0;

  /* data fork extents */

  totablks = file->cat.u.fil.filPyLen / HFS_BLOCKSZ / file->vol->lpa;
  ablkcnt  = 0;
  chunkcnt = 0;

  recp = &file->cat.u.fil.filExtRec;

  while (1)
    {
      ExtDescriptor *newp;

      newp = (ExtDescriptor *) REALLOC(file->dexts, ExtDataRec, ++chunkcnt);
      if (newp == 0)
	{
	  ERROR(ENOMEM, "out of memory");
	  return -1;
	}

      file->dexts = newp;

      memcpy(&file->dexts[file->ndexts], recp, sizeof(ExtDataRec));

      for (nexts = 0; nexts < 3; ++nexts)
	{
	  unsigned int numablks;

	  numablks = file->dexts[file->ndexts].xdrNumABlks;
	  if (numablks)
	    {
	      ++file->ndexts;
	      ablkcnt += numablks;
	    }
	  else
	    break;
	}

      if (ablkcnt < totablks)
	{
	  if (extsearch(file->vol, fkData, file->cat.u.fil.filFlNum,
			ablkcnt, &rec, 0) < 0)
	    return -1;

	  recp = &rec;
	}
      else if (ablkcnt > totablks)
	{
	  ERROR(EIO, "data fork extents exceed file size");
	  return -1;
	}
      else
	break;
    }

  /* resource fork extents */

  totablks = file->cat.u.fil.filRPyLen / HFS_BLOCKSZ / file->vol->lpa;
  ablkcnt  = 0;
  chunkcnt = 0;

  recp = &file->cat.u.fil.filRExtRec;

  while (1)
    {
      ExtDescriptor *newp;

      newp = (ExtDescriptor *) REALLOC(file->rexts, ExtDataRec, ++chunkcnt);
      if (newp == 0)
	{
	  ERROR(ENOMEM, "out of memory");
	  return -1;
	}

      file->rexts = newp;

      memcpy(&file->rexts[file->nrexts], recp, sizeof(ExtDataRec));

      for (nexts = 0; nexts < 3; ++nexts)
	{
	  unsigned int numablks;

	  numablks = file->rexts[file->nrexts].xdrNumABlks;
	  if (numablks)
	    {
	      ++file->nrexts;
	      ablkcnt += numablks;
	    }
	  else
	    break;
	}

      if (ablkcnt < totablks)
	{
	  if (extsearch(file->vol, fkRsrc, file->cat.u.fil.filFlNum,
			ablkcnt, &rec, 0) < 0)
	    return -1;

	  recp = &rec;
	}
      else if (ablkcnt > totablks)
	{
	  ERROR(EIO, "resource fork extents exceed file size");
	  return -1;
	}
      else
	break;
    }

  return 0;
}

/*
 * NAME:	allocate()
 * DESCRIPTION:	reserve disk blocks for a file
 */
static
int allocate(hfsfile *file)
{
  hfsvol *vol;
  block *vbm;
  unsigned int clump, start, search, end, i, *nexts;
  ExtDescriptor **exts, *desc;
  unsigned short *stblk;
  unsigned long *pylen;
  int wrap;

  vol = file->vol;
  vbm = vol->vbm;

  clump = file->cat.u.fil.filClpSize / vol->mdb.drAlBlkSiz;
  start = search = vol->mdb.drAllocPtr;
  end   = vol->mdb.drNmAlBlks;

  if (vol->mdb.drFreeBks < clump)
    {
      ERROR(ENOSPC, "volume full");
      return -1;
    }

  /* search for a contiguous clump */

  wrap = 0;

  while (1)
    {
      if (search + clump > end)
	{
	  search = 0;
	  wrap   = 1;
	}

      if (wrap && search > start)
	{
	  ERROR(ENOSPC, "volume full");
	  return -1;
	}

      for (i = search; i < search + clump; ++i)
	{
	  if (HFS_BMTST(vbm, i))
	    break;
	}

      if (i == search + clump)
	break;  /* found one */

      search = i + 1;
    }

  switch (file->fork)
    {
    case fkRsrc:
      exts  = &file->rexts;
      nexts = &file->nrexts;
      stblk = &file->cat.u.fil.filRStBlk;
      pylen = &file->cat.u.fil.filRPyLen;
      break;

    case fkData:
    default:
      exts  = &file->dexts;
      nexts = &file->ndexts;
      stblk = &file->cat.u.fil.filStBlk;
      pylen = &file->cat.u.fil.filPyLen;
    }

  if (*nexts > 0 &&
      search == (*exts)[*nexts - 1].xdrStABN +
                (*exts)[*nexts - 1].xdrNumABlks)
    (*exts)[*nexts - 1].xdrNumABlks += clump;
  else
    {
      desc = REALLOC(*exts, ExtDescriptor, *nexts + 1);
      if (desc == 0)
	{
	  ERROR(ENOMEM, "out of memory");
	  return -1;
	}

      desc[*nexts].xdrStABN    = search;
      desc[*nexts].xdrNumABlks = clump;

      *exts = desc;
      ++(*nexts);
    }

  /* commit extents changes... */

  if (*nexts == 1)
    *stblk = (*exts)[0].xdrStABN;

  *pylen += clump * vol->mdb.drAlBlkSiz;

  vol->mdb.drAllocPtr = search + clump;
  vol->mdb.drFreeBks -= clump;

  for (i = search; i < search + clump; ++i)
    HFS_BMSET(vbm, i);

  vol->flags |= HFS_UPDATE_MDB | HFS_UPDATE_VBM;

  return 0;
}

/*
 * NAME:	resolve()
 * DESCRIPTION:	translate a pathname; return catalog information
 */
static
int resolve(hfsvol *vol, char *path, CatDataRec *rec,
	    long *parid, char *fname)
{
  long dirid;
  char name[32], *nptr;

  if (*path == 0)
    {
      ERROR(ENOENT, "empty path");
      return -1;
    }
  else if (*path == ':' || strchr(path, ':') == 0)
    {
      dirid = vol->cwd;  /* relative path */

      if (*path == ':')
	++path;

      if (*path == 0)
	{
	  if (getdthread(vol, dirid, rec) < 0)
	    return -1;

	  if (parid)
	    *parid = rec->u.dthd.thdParID;

	  if (catsearch(vol, rec->u.dthd.thdParID,
			rec->u.dthd.thdCName, rec, fname, 0) < 0)
	    return -1;

	  return 0;
	}
    }
  else
    dirid = HFS_CNID_ROOTPAR;  /* absolute path */

  while (1)
    {
      while (*path == ':')
	{
	  ++path;

	  if (getdthread(vol, dirid, rec) < 0)
	    return -1;

	  dirid = rec->u.dthd.thdParID;
	}

      if (*path == 0)
	{
	  if (getdthread(vol, dirid, rec) < 0)
	    return -1;

	  if (parid)
	    *parid = rec->u.dthd.thdParID;

	  if (catsearch(vol, rec->u.dthd.thdParID,
			rec->u.dthd.thdCName, rec, fname, 0) < 0)
	    return -1;

	  return 0;
	}

      nptr = name;
      while (nptr < name + sizeof(name) - 1 && *path && *path != ':')
	*nptr++ = *path++;

      if (*path && *path != ':')
	{
	  ERROR(ENAMETOOLONG, "file name too long");
	  return -1;
	}

      *nptr = 0;
      if (*path == ':')
	++path;

      if (parid)
	*parid = dirid;

      if (catsearch(vol, dirid, name, rec, fname, 0) < 0)
	{
	  if (*path && parid)
	    *parid = 0;

	  if (*path == 0 && fname)
	    strcpy(fname, name);

	  return -1;
	}

      switch (rec->cdrType)
	{
	case cdrDirRec:
	  if (*path == 0)
	    return 0;

	  dirid = rec->u.dir.dirDirID;
	  break;

	case cdrFilRec:
	  if (*path == 0)
	    return 0;

	  ERROR(ENOTDIR, "invalid pathname");
	  return -1;

	default:
	  ERROR(EIO, "unexpected catalog record");
	  return -1;
	}
    }
}

/*
 * NAME:	direntunpack()
 * DESCRIPTION:	unpack catalog information into hfsdirent structure
 */
static
int direntunpack(char *name, CatDataRec *rec, hfsdirent *ent)
{
  strcpy(ent->name, name);

  switch (rec->cdrType)
    {
    case cdrDirRec:
      ent->flags  = HFS_ISDIR;
      ent->cnid   = rec->u.dir.dirDirID;
      ent->crdate = mtoutime(rec->u.dir.dirCrDat);
      ent->mddate = mtoutime(rec->u.dir.dirMdDat);
      ent->dsize  = rec->u.dir.dirVal;
      ent->rsize  = 0;

      ent->type[0]    = 0;
      ent->creator[0] = 0;

      ent->fdflags = rec->u.dir.dirUsrInfo.frFlags;

      return 0;

    case cdrFilRec:
      ent->flags  = (rec->u.fil.filFlags & 0x01) ? HFS_ISLOCKED : 0;
      ent->cnid   = rec->u.fil.filFlNum;
      ent->crdate = mtoutime(rec->u.fil.filCrDat);
      ent->mddate = mtoutime(rec->u.fil.filMdDat);
      ent->dsize  = rec->u.fil.filLgLen;
      ent->rsize  = rec->u.fil.filRLgLen;

      mtoutype(rec->u.fil.filUsrWds.fdType,    ent->type);
      mtoutype(rec->u.fil.filUsrWds.fdCreator, ent->creator);

      ent->fdflags = rec->u.fil.filUsrWds.fdFlags;

      return 0;
    }

  ERROR(EINVAL, "bad catalog entry type");
  return -1;
}

/*
 * NAME:	destruct()
 * DESCRIPTION:	free memory consumed by a volume descriptor
 */
static
void destruct(hfsvol *vol)
{
  FREE(vol->vbm);

  FREE(vol->ext.map);
  if (vol->ext.f.dexts != vol->mdb.drXTExtRec)
    FREE(vol->ext.f.dexts);

  FREE(vol->cat.map);
  if (vol->cat.f.dexts != vol->mdb.drCTExtRec)
    FREE(vol->cat.f.dexts);

  FREE(vol);
}

/*
 * NAME:	flushvol()
 * DESCRIPTION:	flush all pending changes (B*-tree, MDB, VBM) to disk
 */
static
int flushvol(hfsvol *vol)
{
  if (! (vol->flags & HFS_READONLY))
    {
      if ((vol->ext.flags & HFS_UPDATE_BTHDR) &&
	  writebthdr(&vol->ext) < 0)
	return -1;

      if ((vol->cat.flags & HFS_UPDATE_BTHDR) &&
	  writebthdr(&vol->cat) < 0)
	return -1;

      if ((vol->flags & HFS_UPDATE_VBM) &&
	  writevbm(vol) < 0)
	return -1;

      if ((vol->flags & (HFS_UPDATE_MDB | HFS_UPDATE_ALTMDB)) &&
	  writemdb(vol) < 0)
	return -1;
    }

  return 0;
}

/*
 * NAME:	newfolder()
 * DESCRIPTION:	create a new HFS folder
 */
static
int newfolder(hfsvol *vol, long parid, char *name)
{
  CatKeyRec key;
  CatDataRec data;
  long id;
  int i, reclen;
  char *record;

  id = vol->mdb.drNxtCNID++;
  vol->flags |= HFS_UPDATE_MDB;

  /* create directory record */

  data.cdrType   = cdrDirRec;
  data.cdrResrv2 = 0;

  data.u.dir.dirFlags = 0;
  data.u.dir.dirVal   = 0;
  data.u.dir.dirDirID = id;
  data.u.dir.dirCrDat = utomtime(time(0));
  data.u.dir.dirMdDat = data.u.dir.dirCrDat;
  data.u.dir.dirBkDat = 0;

  memset(&data.u.dir.dirUsrInfo,  0, sizeof(data.u.dir.dirUsrInfo));
  memset(&data.u.dir.dirFndrInfo, 0, sizeof(data.u.dir.dirFndrInfo));
  for (i = 0; i < 4; ++i)
    data.u.dir.dirResrv[i] = 0;

  makecatkey(&key, parid, name, 0);

  record = makecatrec(&key, &data, &reclen);
  if (treeinsert(&vol->cat, record, reclen) < 0)
    return -1;

  /* create thread record */

  data.cdrType   = cdrThdRec;
  data.cdrResrv2 = 0;

  data.u.dthd.thdResrv[0] = 0;
  data.u.dthd.thdResrv[1] = 0;
  data.u.dthd.thdParID    = parid;
  strcpy(data.u.dthd.thdCName, name);

  makecatkey(&key, id, "", 0);

  record = makecatrec(&key, &data, &reclen);
  if (treeinsert(&vol->cat, record, reclen) < 0)
    return -1;

  /* update MDB */

  ++vol->mdb.drDirCnt;

  if (parid == HFS_CNID_ROOTDIR)
    ++vol->mdb.drNmRtDirs;

  vol->flags |= HFS_UPDATE_MDB;

  /* update parent directory's valence */

  if (parid != HFS_CNID_ROOTPAR)
    {
      node n;

      if (getdthread(vol, parid, &data) < 0 ||
	  catsearch(vol, data.u.dthd.thdParID, data.u.dthd.thdCName,
		    &data, 0, &n) < 0 ||
	  data.cdrType != cdrDirRec)
	{
	  ERROR(EIO, "can't find parent directory");
	  return -1;
	}

      ++data.u.dir.dirVal;
      data.u.dir.dirMdDat = utomtime(time(0));

      if (putcatrec(&data, &n) < 0)
	return -1;
    }

  return 0;
}

/* High-Level Volume Routines ============================================== */

/*
 * NAME:	hfs_mount()
 * DESCRIPTION:	open an HFS volume; return volume descriptor or 0 (error)
 */
hfsvol *hfs_mount(char *path, int pnum)
{
  hfsvol *vol;
  int i;

  vol = ALLOC(hfsvol, 1);
  if (vol == 0)
    {
      ERROR(ENOMEM, "out of memory");
      return 0;
    }

  vol->flags  = 0;
  vol->vstart = 0;
  vol->vlen   = 0;
  vol->lpa    = 0;
  vol->vbm    = 0;
  vol->cwd    = HFS_CNID_ROOTDIR;
  vol->refs   = 0;

  /* initialize catalog/extents information blocks */
  /* note: the f.dexts fields are initialized only as a bootstrap mechanism */

  vol->ext.f.vol    = vol;
  vol->ext.f.dexts  = vol->mdb.drXTExtRec;
  vol->ext.f.ndexts = 3;
  vol->ext.f.rexts  = 0;
  vol->ext.f.nrexts = 0;
  vol->ext.f.fork   = fkData;
  vol->ext.f.pos    = 0;
  vol->ext.map      = 0;
  vol->ext.mapsz    = 0;
  vol->ext.flags    = 0;
  vol->ext.unpack   = extkeyunpack;
  vol->ext.compare  = extkeycompare;

  strcpy(vol->ext.f.name, "(extents overflow file)");

  vol->cat.f.vol    = vol;
  vol->cat.f.dexts  = vol->mdb.drCTExtRec;
  vol->cat.f.ndexts = 3;
  vol->cat.f.rexts  = 0;
  vol->cat.f.nrexts = 0;
  vol->cat.f.fork   = fkData;
  vol->cat.f.pos    = 0;
  vol->cat.map      = 0;
  vol->cat.mapsz    = 0;
  vol->cat.flags    = 0;
  vol->cat.unpack   = catkeyunpack;
  vol->cat.compare  = catkeycompare;

  strcpy(vol->cat.f.name, "(catalog file)");

  vol->fd = open(path, O_RDWR);
  if (vol->fd < 0 &&
      (errno == EROFS || errno == EACCES))
    {
      vol->flags |= HFS_READONLY;
      vol->fd = open(path, O_RDONLY);
    }

  if (vol->fd < 0)
    {
      destruct(vol);

      ERROR(errno, "error opening device");
      return 0;
    }

  /* find out what kind of media this is */

  if (readblock0(vol, pnum) < 0)
    {
      close(vol->fd);
      destruct(vol);

      return 0;
    }

  /* read MDB and verify this is a Mac HFS volume */

  if (readmdb(vol) < 0 ||
      vol->mdb.drSigWord != 0x4244)
    {
      close(vol->fd);
      destruct(vol);

      ERROR(EINVAL, "not a Macintosh HFS volume");
      return 0;
    }

  /* initialize the B*-tree pseudo-file structures */

  vol->ext.f.cat.cdrType          = cdrFilRec;
  vol->ext.f.cat.u.fil.filFlNum   = HFS_CNID_EXT;
  vol->ext.f.cat.u.fil.filStBlk   = vol->mdb.drXTExtRec[0].xdrStABN;
  vol->ext.f.cat.u.fil.filLgLen   = vol->mdb.drXTFlSize;
  vol->ext.f.cat.u.fil.filPyLen   = vol->mdb.drXTFlSize;
  vol->ext.f.cat.u.fil.filRStBlk  = 0;
  vol->ext.f.cat.u.fil.filRLgLen  = 0;
  vol->ext.f.cat.u.fil.filRPyLen  = 0;
  vol->ext.f.cat.u.fil.filClpSize = vol->mdb.drXTClpSiz;

  memcpy(vol->ext.f.cat.u.fil.filExtRec, vol->mdb.drXTExtRec,
	 sizeof(ExtDataRec));
  for (i = 0; i < 3; ++i)
    {
      vol->ext.f.cat.u.fil.filRExtRec[i].xdrStABN    = 0;
      vol->ext.f.cat.u.fil.filRExtRec[i].xdrNumABlks = 0;
    }

  vol->cat.f.cat.cdrType          = cdrFilRec;
  vol->cat.f.cat.u.fil.filFlNum   = HFS_CNID_CAT;
  vol->cat.f.cat.u.fil.filStBlk   = vol->mdb.drCTExtRec[0].xdrStABN;
  vol->cat.f.cat.u.fil.filLgLen   = vol->mdb.drCTFlSize;
  vol->cat.f.cat.u.fil.filPyLen   = vol->mdb.drCTFlSize;
  vol->cat.f.cat.u.fil.filRStBlk  = 0;
  vol->cat.f.cat.u.fil.filRLgLen  = 0;
  vol->cat.f.cat.u.fil.filRPyLen  = 0;
  vol->cat.f.cat.u.fil.filClpSize = vol->mdb.drCTClpSiz;

  memcpy(vol->cat.f.cat.u.fil.filExtRec, vol->mdb.drCTExtRec,
	 sizeof(ExtDataRec));
  for (i = 0; i < 3; ++i)
    {
      vol->cat.f.cat.u.fil.filRExtRec[i].xdrStABN    = 0;
      vol->cat.f.cat.u.fil.filRExtRec[i].xdrNumABlks = 0;
    }

  /* read the volume bitmap and extents/catalog B*-tree headers */

  if (readvbm(vol) < 0 ||
      readbthdr(&vol->ext) < 0 ||
      readbthdr(&vol->cat) < 0)
    {
      close(vol->fd);
      destruct(vol);
      return 0;
    }

  if (vol->flags & HFS_READONLY)
    vol->mdb.drAtrb |= HFS_ATRB_HLOCKED;
  else
    vol->mdb.drAtrb &= ~HFS_ATRB_HLOCKED;

  vol->mdb.drAtrb &= ~HFS_ATRB_UMOUNTED;  /* set again when unmounted */

  vol->prev = 0;
  vol->next = mounts;

  if (mounts)
    mounts->prev = vol;

  mounts = vol;

  return vol;
}

/*
 * NAME:	hfs_umount()
 * DESCRIPTION:	close an HFS volume
 */
int hfs_umount(hfsvol *vol)
{
  int ecode = 0;

  if (vol->refs)
    {
      ERROR(EBUSY, "volume busy");
      return -1;
    }

  if (vol->flags & HFS_TOUCHED)
    {
      vol->mdb.drAtrb |= HFS_ATRB_UMOUNTED;
      vol->flags |= HFS_UPDATE_MDB;
    }

  if (flushvol(vol) < 0)
    ecode = -1;

  if (close(vol->fd) < 0)
    {
      ERROR(errno, "error closing device");
      ecode = -1;
    }

  if (vol->prev)
    vol->prev->next = vol->next;
  if (vol->next)
    vol->next->prev = vol->prev;
  if (vol == mounts)
    mounts = vol->next;

  destruct(vol);

  return ecode;
}

# if 0
/*
 * NAME:	hfs_umountall()
 * DESCRIPTION:	unmount all mounted volumes
 */
void hfs_umountall(void)
{
  while (mounts)
    {
      hfsvol *vol;

      vol    = mounts;
      mounts = mounts->next;

      hfs_umount(vol);
    }
}
# endif

/*
 * NAME:	hfs_vname()
 * DESCRIPTION:	return the name of an open volume
 */
char *hfs_vname(hfsvol *vol)
{
  return vol->mdb.drVN;
}

/*
 * NAME:	hfs_freebytes()
 * DESCRIPTION:	return the number of free bytes on an open volume
 */
unsigned long hfs_freebytes(hfsvol *vol)
{
  return vol->mdb.drFreeBks * vol->mdb.drAlBlkSiz;
}

/*
 * NAME:	hfs_vcrdate()
 * DESCRIPTION:	return time of volume creation
 */
long hfs_vcrdate(hfsvol *vol)
{
  return mtoutime(vol->mdb.drCrDate);
}

/*
 * NAME:	hfs_vmddate()
 * DESCRIPTION:	return time of last volume modification
 */
long hfs_vmddate(hfsvol *vol)
{
  return mtoutime(vol->mdb.drLsMod);
}

/*
 * NAME:	valid_offset()
 * DESCRIPTION:	return 1 iff the given offset is valid
 */
static
int valid_offset(int fd, unsigned long offs)
{
  char c;

  if (lseek(fd, offs, SEEK_SET) < 0)
    return 0;

  if (read(fd, &c, 1) < 1)
    return 0;

  return 1;
}

/*
 * NAME:	hfs_format()
 * DESCRIPTION:	write a new filesystem
 */
int hfs_format(char *path, int pnum, char *vname)
{
  hfsvol vol;
  int vbmsz;
  int i, ecode = 0;
  block vbm[16];
  char emap[HFS_MAP1SZ], cmap[HFS_MAP1SZ];

  if (strchr(vname, ':'))
    {
      ERROR(EINVAL, "volume name may not contain colons");
      return -1;
    }

  i = strlen(vname);
  if (i < 1 || i > 27)
    {
      ERROR(EINVAL, "volume name must be 1-27 chars");
      return -1;
    }

  vol.flags  = 0;
  vol.vstart = 0;
  vol.vlen   = 0;
  vol.lpa    = 0;
  vol.vbm    = vbm;
  vol.cwd    = HFS_CNID_ROOTDIR;
  vol.refs   = 0;

  vol.ext.f.vol    = &vol;
  vol.ext.f.dexts  = 0;
  vol.ext.f.ndexts = 0;
  vol.ext.f.rexts  = 0;
  vol.ext.f.nrexts = 0;
  vol.ext.f.fork   = fkData;
  vol.ext.f.pos    = 0;
  vol.ext.map      = emap;
  vol.ext.mapsz    = sizeof(emap);
  vol.ext.flags    = 0;
  vol.ext.unpack   = extkeyunpack;
  vol.ext.compare  = extkeycompare;

  strcpy(vol.ext.f.name, "(extents overflow file)");

  vol.cat.f.vol    = &vol;
  vol.cat.f.dexts  = 0;
  vol.cat.f.ndexts = 0;
  vol.cat.f.rexts  = 0;
  vol.cat.f.nrexts = 0;
  vol.cat.f.fork   = fkData;
  vol.cat.f.pos    = 0;
  vol.cat.map      = cmap;
  vol.cat.mapsz    = sizeof(cmap);
  vol.cat.flags    = 0;
  vol.cat.unpack   = catkeyunpack;
  vol.cat.compare  = catkeycompare;

  strcpy(vol.cat.f.name, "(catalog file)");

  vol.fd = open(path, O_RDWR);
  if (vol.fd < 0)
    {
      ERROR(errno, "error opening device");
      return -1;
    }

  if (pnum > 0)
    {
      if (readpm(&vol, pnum) < 0)
	{
	  close(vol.fd);
	  return -1;
	}
    }
  else  /* determine size of entire device */
    {
      unsigned long low, high, mid;

      for (low = 0, high = 800 * 1024; valid_offset(vol.fd, high); high *= 2)
	low = high;

      while (low < high - 1)
	{
	  mid = (low + high) / 2;

	  if (valid_offset(vol.fd, mid))
	    low = mid;
	  else
	    high = mid;
	}

      vol.vlen = (low + 1) / HFS_BLOCKSZ;
    }

  if (vol.vlen < 800 * 1024 / HFS_BLOCKSZ)
    {
      close(vol.fd);

      ERROR(EINVAL, "volume size must be >= 800K");
      return -1;
    }

  /* initialize volume geometry */

  vol.lpa = 1 + vol.vlen / 65536;

  vbmsz = (vol.vlen / vol.lpa + 4095) / 4096;

  vol.mdb.drSigWord  = 0x4244;
  vol.mdb.drCrDate   = utomtime(time(0));
  vol.mdb.drLsMod    = vol.mdb.drCrDate;
  vol.mdb.drAtrb     = 0;
  vol.mdb.drNmFls    = 0;
  vol.mdb.drVBMSt    = 3;
  vol.mdb.drAllocPtr = 0;
  vol.mdb.drNmAlBlks = (vol.vlen - 5 - vbmsz) / vol.lpa;
  vol.mdb.drAlBlkSiz = vol.lpa * HFS_BLOCKSZ;
  vol.mdb.drClpSiz   = vol.mdb.drAlBlkSiz * 4;
  vol.mdb.drAlBlSt   = 3 + vbmsz;
  vol.mdb.drNxtCNID  = HFS_CNID_ROOTDIR;  /* modified later */
  vol.mdb.drFreeBks  = vol.mdb.drNmAlBlks;

  strcpy(vol.mdb.drVN, vname);

  vol.mdb.drVolBkUp  = 0;
  vol.mdb.drVSeqNum  = 0;
  vol.mdb.drWrCnt    = 0;
  vol.mdb.drXTClpSiz = vol.vlen / 128 * HFS_BLOCKSZ;
  vol.mdb.drCTClpSiz = vol.vlen / 128 * HFS_BLOCKSZ;
  vol.mdb.drNmRtDirs = 0;
  vol.mdb.drFilCnt   = 0;
  vol.mdb.drDirCnt   = -1;  /* incremented when root folder is created */

  for (i = 0; i < 8; ++i)
    vol.mdb.drFndrInfo[i] = 0;

  vol.mdb.drVCSize   = 0;
  vol.mdb.drVCBMSize = 0;
  vol.mdb.drCtlCSize = 0;

  vol.mdb.drXTFlSize = 0;
  for (i = 0; i < 3; ++i)
    {
      vol.mdb.drXTExtRec[i].xdrStABN    = 0;
      vol.mdb.drXTExtRec[i].xdrNumABlks = 0;
    }

  vol.mdb.drCTFlSize = 0;
  for (i = 0; i < 3; ++i)
    {
      vol.mdb.drCTExtRec[i].xdrStABN    = 0;
      vol.mdb.drCTExtRec[i].xdrNumABlks = 0;
    }

  /* initialize volume bitmap */

  memset(vol.vbm, 0, sizeof(vbm));

  /* create extents overflow file */

  vol.ext.f.cat.cdrType          = cdrFilRec;
  vol.ext.f.cat.u.fil.filFlNum   = HFS_CNID_EXT;
  vol.ext.f.cat.u.fil.filStBlk   = vol.mdb.drXTExtRec[0].xdrStABN;
  vol.ext.f.cat.u.fil.filLgLen   = vol.mdb.drXTFlSize;
  vol.ext.f.cat.u.fil.filPyLen   = vol.mdb.drXTFlSize;
  vol.ext.f.cat.u.fil.filRStBlk  = 0;
  vol.ext.f.cat.u.fil.filRLgLen  = 0;
  vol.ext.f.cat.u.fil.filRPyLen  = 0;
  vol.ext.f.cat.u.fil.filClpSize = vol.mdb.drXTClpSiz;

  if (allocate(&vol.ext.f) < 0)
    ecode = -1;

  vol.mdb.drXTFlSize = vol.ext.f.cat.u.fil.filPyLen;

  vol.ext.hdr.bthDepth    = 0;
  vol.ext.hdr.bthRoot     = 0;
  vol.ext.hdr.bthNRecs    = 0;
  vol.ext.hdr.bthFNode    = 0;
  vol.ext.hdr.bthLNode    = 0;
  vol.ext.hdr.bthNodeSize = HFS_BLOCKSZ;
  vol.ext.hdr.bthKeyLen   = 0x07;
  vol.ext.hdr.bthNNodes   = vol.mdb.drXTFlSize / vol.mdb.drAlBlkSiz;
  vol.ext.hdr.bthFree     = vol.ext.hdr.bthNNodes;

  for (i = 0; i < 76; ++i)
    vol.ext.hdr.bthResv[i] = 0;

  vol.ext.hdrnd.bt   = &vol.ext;
  vol.ext.hdrnd.nnum = 0;

  vol.ext.hdrnd.nd.ndFLink   = 0;
  vol.ext.hdrnd.nd.ndBLink   = 0;
  vol.ext.hdrnd.nd.ndType    = ndHdrNode;
  vol.ext.hdrnd.nd.ndNHeight = 0;

  memset(vol.ext.map, 0, vol.ext.mapsz);

  if (newnode(&vol.ext.hdrnd) < 0)
    ecode = -1;

  vol.ext.hdrnd.roff[1] = 0x078;
  vol.ext.hdrnd.roff[2] = 0x0f8;
  vol.ext.hdrnd.roff[3] = 0x1f8;

  memset(HFS_NODEREC(vol.ext.hdrnd, 1), 0, 128);

  vol.ext.flags |= HFS_UPDATE_BTHDR;

  /* create catalog file */

  vol.cat.f.cat.cdrType          = cdrFilRec;
  vol.cat.f.cat.u.fil.filFlNum   = HFS_CNID_CAT;
  vol.cat.f.cat.u.fil.filStBlk   = vol.mdb.drCTExtRec[0].xdrStABN;
  vol.cat.f.cat.u.fil.filLgLen   = vol.mdb.drCTFlSize;
  vol.cat.f.cat.u.fil.filPyLen   = vol.mdb.drCTFlSize;
  vol.cat.f.cat.u.fil.filRStBlk  = 0;
  vol.cat.f.cat.u.fil.filRLgLen  = 0;
  vol.cat.f.cat.u.fil.filRPyLen  = 0;
  vol.cat.f.cat.u.fil.filClpSize = vol.mdb.drCTClpSiz;

  if (allocate(&vol.cat.f) < 0)
    ecode = -1;

  vol.mdb.drCTFlSize = vol.cat.f.cat.u.fil.filPyLen;

  vol.cat.hdr.bthDepth    = 0;
  vol.cat.hdr.bthRoot     = 0;
  vol.cat.hdr.bthNRecs    = 0;
  vol.cat.hdr.bthFNode    = 0;
  vol.cat.hdr.bthLNode    = 0;
  vol.cat.hdr.bthNodeSize = HFS_BLOCKSZ;
  vol.cat.hdr.bthKeyLen   = 0x25;
  vol.cat.hdr.bthNNodes   = vol.mdb.drCTFlSize / vol.mdb.drAlBlkSiz;
  vol.cat.hdr.bthFree     = vol.ext.hdr.bthNNodes;

  for (i = 0; i < 76; ++i)
    vol.cat.hdr.bthResv[i] = 0;

  vol.cat.hdrnd.bt   = &vol.cat;
  vol.cat.hdrnd.nnum = 0;

  vol.cat.hdrnd.nd.ndFLink   = 0;
  vol.cat.hdrnd.nd.ndBLink   = 0;
  vol.cat.hdrnd.nd.ndType    = ndHdrNode;
  vol.cat.hdrnd.nd.ndNHeight = 0;

  memset(vol.cat.map, 0, vol.cat.mapsz);

  if (newnode(&vol.cat.hdrnd) < 0)
    ecode = -1;

  vol.cat.hdrnd.roff[1] = 0x078;
  vol.cat.hdrnd.roff[2] = 0x0f8;
  vol.cat.hdrnd.roff[3] = 0x1f8;

  memset(HFS_NODEREC(vol.cat.hdrnd, 1), 0, 128);

  vol.cat.flags |= HFS_UPDATE_BTHDR;

  /* create root folder */

  if (newfolder(&vol, HFS_CNID_ROOTPAR, vname) < 0)
    ecode = -1;

  vol.mdb.drNxtCNID = 16;

  /* finish up */

  if (ecode == 0)
    {
      block b;

      /* write boot blocks */

      memset(&b, 0, sizeof(b));
      writelblock(&vol, 0, &b);
      writelblock(&vol, 1, &b);

      /* flush other disk state */

      if (vol.ext.f.ndexts > 3 ||
	  vol.cat.f.ndexts > 3)
	abort();

      memcpy(vol.mdb.drXTExtRec, vol.ext.f.dexts,
	     SIZE(ExtDescriptor, vol.ext.f.ndexts));
      memcpy(vol.mdb.drCTExtRec, vol.cat.f.dexts,
	     SIZE(ExtDescriptor, vol.cat.f.ndexts));

      vol.mdb.drAtrb |= HFS_ATRB_UMOUNTED;
      vol.flags |= HFS_UPDATE_MDB | HFS_UPDATE_ALTMDB | HFS_UPDATE_VBM;

      if (flushvol(&vol) < 0)
	ecode = -1;
    }

  if (close(vol.fd) < 0)
    {
      ERROR(errno, "error closing device");
      ecode = -1;
    }

  FREE(vol.ext.f.dexts);
  FREE(vol.cat.f.dexts);

  return ecode;
}

/* High-Level Directory Routines =========================================== */

/*
 * NAME:	hfs_chdir()
 * DESCRIPTION:	change current HFS directory
 */
int hfs_chdir(hfsvol *vol, char *path)
{
  CatDataRec rec;

  if (resolve(vol, path, &rec, 0, 0) < 0)
    return -1;

  if (rec.cdrType != cdrDirRec)
    {
      ERROR(ENOTDIR, "not a directory");
      return -1;
    }

  vol->cwd = rec.u.dir.dirDirID;

  return 0;
}

# if 0
/*
 * NAME:	hfs_getcwd()
 * DESCRIPTION:	return full path (up to 255 chars) to current HFS directory
 */
char *hfs_getcwd(hfsvol *vol)
{
  static char path[256];
  char *ptr;
  long cwd = vol->cwd;

  /* this should really be implemented as a client routine */

  path[255] = 0;
  ptr = &path[255];

  while (cwd != HFS_CNID_ROOTPAR)
    {
      CatDataRec thread;
      int len;

      if (getdthread(vol, cwd, &thread) < 0)
	return 0;

      len = strlen(thread.u.dthd.thdCName);

      if (ptr - len - 1 < path)
	{
	  ERROR(ENAMETOOLONG, "path too long");
	  return 0;
	}

      if (ptr < &path[255])
	*--ptr = ':';

      ptr -= len;
      memcpy(ptr, thread.u.dthd.thdCName, len);

      cwd = thread.u.dthd.thdParID;
    }

  return ptr;
}
# endif

/*
 * NAME:	hfs_cwdid()
 * DESCRIPTION:	return the current working directory ID
 */
long hfs_cwdid(hfsvol *vol)
{
  return vol->cwd;
}

/*
 * NAME:	hfs_dirinfo()
 * DESCRIPTION:	given a directory ID, return its (name and) parent ID
 */
int hfs_dirinfo(hfsvol *vol, long *id, char *name)
{
  CatDataRec thread;

  if (getdthread(vol, *id, &thread) < 0)
    return -1;

  *id = thread.u.dthd.thdParID;

  if (name)
    strcpy(name, thread.u.dthd.thdCName);

  return 0;
}

/*
 * NAME:	hfs_opendir()
 * DESCRIPTION:	prepare to read the contents of a directory
 */
hfsdir *hfs_opendir(hfsvol *vol, char *path)
{
  hfsdir *dir;
  CatKeyRec key;
  CatDataRec rec;
  long dirid;

  if (resolve(vol, path, &rec, 0, 0) < 0)
    return 0;

  if (rec.cdrType != cdrDirRec)
    {
      ERROR(ENOTDIR, "not a directory");
      return 0;
    }

  dirid = rec.u.dir.dirDirID;

  dir = ALLOC(hfsdir, 1);
  if (dir == 0)
    {
      ERROR(ENOMEM, "out of memory");
      return 0;
    }

  dir->dirid = dirid;
  dir->n.bt  = &vol->cat;

  makecatkey(&key, dirid, "", 0);
  if (treesearch(&dir->n, (char *) &key) < 0)
    {
      FREE(dir);
      return 0;
    }

  ++vol->refs;

  return dir;
}

/*
 * NAME:	hfs_readdir()
 * DESCRIPTION:	return the next entry in the directory
 */
int hfs_readdir(hfsdir *dir, hfsdirent *ent)
{
  CatKeyRec *key;
  CatDataRec rec;
  char *ptr;

  if (dir->n.rnum == -1)
    {
      ERROR(ENOENT, "no more entries");
      return -1;
    }

  while (1)
    {
      ++dir->n.rnum;

      while (dir->n.rnum >= dir->n.nd.ndNRecs)
	{
	  dir->n.nnum = dir->n.nd.ndFLink;

	  if (getnode(&dir->n) < 0)
	    {
	      dir->n.rnum = -1;
	      return -1;
	    }

	  dir->n.rnum = 0;
	}

      ptr = HFS_NODEREC(dir->n, dir->n.rnum);
      key = (CatKeyRec *) catkeyunpack(ptr);

      if (key->ckrParID != dir->dirid)
	{
	  dir->n.rnum = -1;

	  ERROR(ENOENT, "no more entries");
	  return -1;
	}

      catdataunpack(HFS_RECDATA(ptr), &rec);

      switch (rec.cdrType)
	{
	case cdrDirRec:
	case cdrFilRec:
	  if (direntunpack(key->ckrCName, &rec, ent) < 0)
	    return -1;

	  return 0;

	case cdrThdRec:
	case cdrFThdRec:
	  break;

	default:
	  dir->n.rnum = -1;

	  ERROR(EIO, "unexpected directory entry found");
	  return -1;
	}
    }
}

/*
 * NAME:	hfs_closedir()
 * DESCRIPTION:	stop reading a directory
 */
int hfs_closedir(hfsdir *dir)
{
  --dir->n.bt->f.vol->refs;

  FREE(dir);

  return 0;
}

/* High-Level File Routines ================================================ */

/*
 * NAME:	hfs_open()
 * DESCRIPTION:	prepare a file for I/O
 */
hfsfile *hfs_open(hfsvol *vol, char *path)
{
  hfsfile *file;

  file = ALLOC(hfsfile, 1);
  if (file == 0)
    {
      ERROR(ENOMEM, "out of memory");
      return 0;
    }

  if (resolve(vol, path, &file->cat, 0, file->name) < 0)
    {
      FREE(file);
      return 0;
    }

  if (file->cat.cdrType != cdrFilRec)
    {
      FREE(file);
      ERROR(EISDIR, "not a file");
      return 0;
    }

  file->vol   = vol;
  file->fork  = fkData;
  file->pos   = 0;

  if (getextents(file) < 0)
    {
      FREE(file);
      return 0;
    }

  ++vol->refs;

  return file;
}

/*
 * NAME:	hfs_fork()
 * DESCRIPTION:	select file fork for I/O operations
 */
void hfs_fork(hfsfile *file, int fork)
{
  file->fork = fork ? fkRsrc : fkData;
  file->pos  = 0;
}

/*
 * NAME:	hfs_read()
 * DESCRIPTION:	read from an open file
 */
long hfs_read(hfsfile *file, char *ptr, unsigned long len)
{
  unsigned long flen, count;

  flen = (file->fork == fkData) ?
    file->cat.u.fil.filLgLen : file->cat.u.fil.filRLgLen;

  if (file->pos + len > flen)
    len = flen - file->pos;

  count = len;
  while (count)
    {
      block b;
      unsigned long bnum, offs, chunk;

      bnum = file->pos / HFS_BLOCKSZ;
      offs = file->pos % HFS_BLOCKSZ;

      if (getfblock(file, bnum, &b) < 0)
	return -1;

      chunk = HFS_BLOCKSZ - offs;
      if (chunk > count)
	chunk = count;

      memcpy(ptr, (char *) &b + offs, chunk);
      ptr += chunk;

      file->pos += chunk;
      count     -= chunk;
    }

  return len;
}

/*
 * NAME:	hfs_write()
 * DESCRIPTION:	write to an open file
 */
long hfs_write(hfsfile *file, char *ptr, unsigned long len)
{
# if 0
  unsigned long flen, count;
# endif

  if (file->vol->flags & HFS_READONLY)
    {
      ERROR(EROFS, "read-only filesystem");
      return -1;
    }

  ERROR(ENOSYS, "not yet implemented");
  return -1;

# if 0
  flen = (file->fork == fkData) ?
    file->cat.u.fil.filLgLen : file->cat.u.fil.filRLgLen;

  count = len;
  while (count)
    {
      block b;
      unsigned long bnum, offs, chunk;

      bnum = file->pos / HFS_BLOCKSZ;
      offs = file->pos % HFS_BLOCKSZ;

      if (getfblock(file, bnum, &b) < 0)
	return -1;
    }

  return len;
# endif
}

/*
 * NAME:	hfs_lseek()
 * DESCRIPTION:	change file seek pointer
 */
long hfs_lseek(hfsfile *file, long offset, int from)
{
  unsigned long fsize;
  long newpos;

  fsize = (file->fork == fkData) ?
    file->cat.u.fil.filLgLen : file->cat.u.fil.filRLgLen;

  switch (from)
    {
    case SEEK_SET:
      newpos = offset;
      break;

    case SEEK_CUR:
      newpos = file->pos + offset;
      break;

    case SEEK_END:
      newpos = fsize + offset;
      break;

    default:
      ERROR(EINVAL, "invalid argument");
      return -1;
    }

  if (newpos < 0)
    newpos = 0;
  else if (newpos > fsize)
    newpos = fsize;

  file->pos = newpos;

  return newpos;
}

/*
 * NAME:	hfs_close()
 * DESCRIPTION:	close a file
 */
int hfs_close(hfsfile *file)
{
  hfsvol *vol = file->vol;

  FREE(file);

  --vol->refs;

  return flushvol(vol);
}

/* High-Level Catalog Routines ============================================= */

/*
 * NAME:	hfs_stat()
 * DESCRIPTION:	return catalog information for an arbitrary path
 */
int hfs_stat(hfsvol *vol, char *path, hfsdirent *ent)
{
  CatDataRec rec;
  char name[32];

  if (resolve(vol, path, &rec, 0, name) < 0)
    return -1;

  return direntunpack(name, &rec, ent);
}

/*
 * NAME:	hfs_fstat()
 * DESCRIPTION:	return catalog information for an open file
 */
int hfs_fstat(hfsfile *file, hfsdirent *ent)
{
  return direntunpack(file->name, &file->cat, ent);
}

/*
 * NAME:	hfs_mkdir()
 * DESCRIPTION:	create a new directory
 */
int hfs_mkdir(hfsvol *vol, char *path)
{
  CatDataRec rec;
  long parid;
  char name[32];

  if (vol->flags & HFS_READONLY)
    {
      ERROR(EROFS, "read-only filesystem");
      return -1;
    }

  if (resolve(vol, path, &rec, &parid, name) >= 0)
    {
      ERROR(EEXIST, "file exists");
      return -1;
    }

  if (errno != ENOENT || parid == 0)
    return -1;

  if (newfolder(vol, parid, name) < 0)
    return -1;

  return 0;
}

/*
 * NAME:	hfs_rmdir()
 * DESCRIPTION:	delete an empty directory
 */
int hfs_rmdir(hfsvol *vol, char *path)
{
  CatDataRec rec;
  long parid;

  if (vol->flags & HFS_READONLY)
    {
      ERROR(EROFS, "read-only filesystem");
      return -1;
    }

  if (resolve(vol, path, &rec, 0, 0) < 0)
    return -1;

  if (rec.cdrType != cdrDirRec)
    {
      ERROR(ENOTDIR, "not a directory");
      return -1;
    }

  if (rec.u.dir.dirVal != 0)
    {
      ERROR(ENOTEMPTY, "directory not empty");
      return -1;
    }

  ERROR(ENOSYS, "not yet implemented");
  return -1;

  --vol->mdb.drDirCnt;
  if (parid == HFS_CNID_ROOTDIR)
    --vol->mdb.drNmRtDirs;

  vol->flags |= HFS_UPDATE_MDB;
}

/*
 * NAME:	hfs_create()
 * DESCRIPTION:	create a new file
 */
int hfs_create(hfsvol *vol, char *path, char *type, char *creator)
{
  CatDataRec rec;
  long parid;
  char name[32];

  if (vol->flags & HFS_READONLY)
    {
      ERROR(EROFS, "read-only filesystem");
      return -1;
    }

  if (resolve(vol, path, &rec, &parid, name) >= 0)
    {
      ERROR(EEXIST, "file exists");
      return -1;
    }

  if (errno != ENOENT || parid == 0)
    return -1;

  /* create file `name' in parent `parid' */

  ERROR(ENOSYS, "not yet implemented");
  return -1;

  ++vol->mdb.drFilCnt;
  if (parid == HFS_CNID_ROOTDIR)
    ++vol->mdb.drNmFls;

  vol->flags |= HFS_UPDATE_MDB;
}

/*
 * NAME:	hfs_delete()
 * DESCRIPTION:	remove both forks of a file
 */
int hfs_delete(hfsvol *vol, char *path)
{
  CatDataRec rec;
  long parid;

  if (vol->flags & HFS_READONLY)
    {
      ERROR(EROFS, "read-only filesystem");
      return -1;
    }

  if (resolve(vol, path, &rec, 0, 0) < 0)
    return -1;

  if (rec.cdrType != cdrFilRec)
    {
      ERROR(EISDIR, "is a directory");
      return -1;
    }

  ERROR(ENOSYS, "not yet implemented");
  return -1;

  --vol->mdb.drFilCnt;
  if (parid == HFS_CNID_ROOTDIR)
    --vol->mdb.drNmFls;

  vol->flags |= HFS_UPDATE_MDB;
}

/*
 * NAME:	hfs_rename()
 * DESCRIPTION:	change the name of a file or directory or volume
 */
int hfs_rename(hfsvol *vol, char *path, char *newname)
{
  CatDataRec rec;

  if (vol->flags & HFS_READONLY)
    {
      ERROR(EROFS, "read-only filesystem");
      return -1;
    }

  if (resolve(vol, path, &rec, 0, 0) < 0)
    return -1;

  ERROR(ENOSYS, "not yet implemented");
  return -1;
}

/*
 * NAME:	hfs_move()
 * DESCRIPTION:	change the location of a file or directory
 */
int hfs_move(hfsvol *vol, char *path, char *newdir)
{
  CatDataRec src, dst;
  long dstid;

  if (vol->flags & HFS_READONLY)
    {
      ERROR(EROFS, "read-only filesystem");
      return -1;
    }

  if (resolve(vol, path, &src, 0, 0) < 0)
    return -1;
  if (resolve(vol, newdir, &dst, 0, 0) < 0)
    return -1;

  if (dst.cdrType != cdrDirRec)
    {
      ERROR(ENOTDIR, "not a directory");
      return -1;
    }

  dstid = dst.u.dir.dirDirID;

  if (src.cdrType == cdrDirRec)
    {
      long id;
      long srcid = src.u.dir.dirDirID;
      CatDataRec thread;

      /* can't move root directory anywhere */

      if (srcid == HFS_CNID_ROOTDIR)
	{
	  ERROR(EINVAL, "illegal move");
	  return -1;
	}

      /* make sure we aren't trying to move directory inside itself */

      for (id = dstid; id != HFS_CNID_ROOTDIR; id = thread.u.dthd.thdParID)
	{
	  if (id == srcid)
	    {
	      ERROR(EINVAL, "illegal move");
	      return -1;
	    }

	  if (getdthread(vol, id, &thread) < 0)
	    return -1;
	}
    }

  ERROR(ENOSYS, "not yet implemented");
  return -1;

  /* update MDB stats if something moves in/out of root dir */
}
