# include <stdio.h>
# include <string.h>
# include <time.h>

# include "hfs.h"
# include "hcwd.h"
# include "hfsutil.h"
# include "version.h"

# include "hmount.h"
# include "hvol.h"
# include "hdir.h"
# include "hcd.h"
# include "hpwd.h"
# include "hcat.h"
# include "hformat.h"

int main(int argc, char *argv[])
{
  int i;
  char *argv0;

  struct {
    char *name;
    int (*func)(int, char *[]);
  } list[] = {
    { "hmount",  hmount_main  },
    { "hvol",    hvol_main    },
    { "hdir",    hdir_main    },
    { "hcd",     hcd_main     },
    { "hpwd",    hpwd_main    },
    { "hcat",    hcat_main    },
    { "hformat", hformat_main },
    { 0,        0             }
  };

  if (argc > 1)
    {
      if (strcmp(argv[1], "--version") == 0)
	{
	  printf("%s - %s\n", version, copyright);
	  printf("`%s --license' for licensing information.\n", argv[0]);
	  return 0;
	}
      else if (strcmp(argv[1], "--license") == 0)
	{
	  printf("\n%s", license);
	  return 0;
	}
    }

  argv0 = strrchr(argv[0], '/');
  if (argv0 == 0)
    argv0 = argv[0];
  else
    ++argv0;

  for (i = 0; list[i].name; ++i)
    {
      if (strcmp(argv0, list[i].name) == 0)
	{
	  int ret;

	  if (hcwd_init() < 0)
	    {
	      perror("Failed to initialize HFS working directories");
	      return 1;
	    }

	  ret = list[i].func(argc, argv);

	  if (hcwd_finish() < 0)
	    {
	      perror("Failed to save working directory state");
	      return 1;
	    }

	  return ret;
	}
    }

  fprintf(stderr, "Unknown operation `%s'\n", argv0);
  return 1;
}

void hfs_perror(char *msg)
{
  fprintf(stderr, "%s: %s\n", msg, hfs_error);
}

hfsvol *hfs_remount(char *progname, mountent *ent)
{
  hfsvol *vol;

  if (ent == 0)
    {
      fprintf(stderr, "%s: No volume is current; use `hmount' or `hvol'\n",
	      progname);
      return 0;
    }

  vol = hfs_mount(ent->path, ent->partno);
  if (vol == 0)
    {
      hfs_perror(ent->path);
      return 0;
    }

  if (strcmp(hfs_vname(vol), ent->vname) != 0)
    {
      hfs_umount(vol);
      fprintf(stderr,
	      "%s: Expected volume not found; replace media or use `hmount'\n",
	      progname);
      return 0;
    }

  if (hfs_chdir(vol, ent->cwd) < 0)
    {
      hfs_umount(vol);
      hfs_perror("Can't find current HFS directory");
      return 0;
    }

  return vol;
}

void hfs_pinfo(hfsvol *vol)
{
  long date;

  printf("Volume name is \"%s\"\n", hfs_vname(vol));
  date = hfs_vcrdate(vol);
  printf("Volume was created on %s", ctime(&date));
  date = hfs_vmddate(vol);
  printf("Volume was last modified on %s", ctime(&date));
  printf("Volume has %lu bytes free\n", hfs_freebytes(vol));
}
