#!./hfssh

###############################################################################
#
# hfsutils - tools for reading and writing Macintosh HFS volumes
# Copyright (C) 1996 Robert Leslie
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
###############################################################################

proc hinfo {} {
    global vol

    if {! [info exists vol]} {
	error "nothing mounted"
    }

    puts "Volume name is \"[$vol vname]\""
    puts "Volume was created on [ctime [$vol vcrdate]]"
    puts "Volume was last modified on [ctime [$vol vmddate]]"
    puts "Volume has [$vol freebytes] free bytes"
}

proc hmount {path {partno 1}} {
    global vol dev

    if {[info exists vol]} {
	error "$dev already mounted"
    }

    set vol [hfs mount $path $partno]
    set dev $path

    hinfo
}

proc humount {} {
    global vol dev

    if {! [info exists vol]} {
	error "nothing mounted"
    }

    $vol umount

    unset vol dev
}

proc hpwd {} {
    global vol

    if {! [info exists vol]} {
	error "nothing mounted"
    }

    set cwd [$vol cwdid]

    while {$cwd != 1} {
	set info [$vol dirinfo $cwd]

	if {[info exists path]} {
	    set path "[lindex $info 0]:$path"
	} else {
	    set path [lindex $info 0]
	}

	set cwd [lindex $info 1]
    }

    puts $path
}

proc hcd {{path ""}} {
    global vol

    if {! [info exists vol]} {
	error "nothing mounted"
    }

    if {[string length $path] == 0} {
	set path "[$vol vname]:"
    }

    $vol cd $path
}

proc assoc {list var} {
    upvar $var array

    foreach elt $list {
	if {[info exists key]} {
	    set array($key) $elt
	    unset key
	} else {
	    set key $elt
	}
    }
}

proc timestr {secs} {
    set ctime [ctime $secs]

    return "[string range $ctime 4 15][string range $ctime 19 23]"
}

proc plural {what yes no} {
    if {$what != 1} {
	return $yes
    } else {
	return $no
    }
}

proc hdir {{path ":"}} {
    global vol

    if {! [info exists vol]} {
	error "nothing mounted"
    }

    foreach ent [$vol dir $path] {
	assoc $ent item

	if {$item(kind) == "directory"} {
	    puts [format "d %9lu item%s               %s %s"  \
		    $item(size)  \
		    [plural $item(size) "s" " "]  \
		    [timestr $item(mddate)]  \
		    $item(name)]
	} else {
	    puts [format "f %4s/%4s %9lu %9lu %s %s"  \
		    $item(type)  \
		    $item(creator)  \
		    $item(rsize)  \
		    $item(dsize)  \
		    [timestr $item(mddate)]  \
		    $item(name)]
	}
    }
}

proc hstat {{path ":"}} {
    global vol

    if {! [info exists vol]} {
	error "nothing mounted"
    }

    assoc [$vol stat $path] item

    foreach elt [array names item] {
	if {[regexp {date$} $elt]} {
	    set value [ctime $item($elt)]
	} else {
	    set value $item($elt)
	}

	puts [format "%-10s %s" "$elt:" $value]
    }
}

proc hmkdir {path} {
    global vol

    if {! [info exists vol]} {
	error "nothing mounted"
    }

    $vol mkdir $path
}

proc hrmdir {path} {
    global vol

    if {! [info exists vol]} {
	error "nothing mounted"
    }

    $vol rmdir $path
}

proc hcreate {path {type "TEXT"} {creator "UNIX"}} {
    global vol

    if {! [info exists vol]} {
	error "nothing mounted"
    }

    $vol create $path $type $creator
}

proc hdelete {path} {
    global vol

    if {! [info exists vol]} {
	error "nothing mounted"
    }

    $vol delete $path
}

proc hrename {path newname} {
    global vol

    if {! [info exists vol]} {
	error "nothing mounted"
    }

    $vol rename $path $newname
}

proc hmove {path newdir} {
    global vol

    if {! [info exists vol]} {
	error "nothing mounted"
    }

    $vol move $path $newdir
}

proc hcat {path} {
    global vol

    if {! [info exists vol]} {
	error "nothing mounted"
    }

    set file [$vol open $path]

    while {1} {
	set buf [$file read 512]
	if {[string length $buf] == 0} {
	    $file close
	    break
	}

	regsub -all "\r" $buf "\n" buf

	puts -nonewline $buf
    }
}

proc hcopyout {path {dest "."} {mode "macb"}} {
    global vol

    if {! [info exists vol]} {
	error "nothing mounted"
    }

    $vol copyout $mode $path $dest
}

proc hformat {path {vname "Untitled"}} {
    global vol dev

    if {[info exists dev]} {
	humount
    }

    set partno 0
    hfs format $path $partno $vname

    if {! [info exists vol]} {
	hmount $path $partno
    }
}

###############################################################################

proc license {} {
    puts -nonewline "\n[hfs license]"
}

if {[string compare [lindex $argv 0] "--license"] == 0} {
    license
    exit
}

puts "[hfs version] - [hfs copyright]"

if {[string compare [lindex $argv 0] "--version"] == 0} {
    puts "`$argv0 --license' for licensing information."
    exit
}

puts "This is free software but comes with ABSOLUTELY NO WARRANTY."
puts "Type `license' for details.\n"

###############################################################################

rename exit _exit

proc exit {{code 0}} {
    global vol

    if {[info exists vol]} {
	$vol umount
    }

    _exit $code
}

###############################################################################

if {$argc > 0} {
    eval hmount $argv
}

while {1} {
    puts -nonewline "hfs> "
    if {[gets stdin line] == -1} {
	exit
    }

    while {! [info complete $line]} {
	if {[gets stdin more] == -1} {
	    break
	} else {
	    set line "$line$more"
	}
    }

    if {[string compare $line "quit"] == 0} {
	exit
    }

    if {[string length [info procs "h[lindex $line 0]"]] > 0} {
	set result [catch {eval h$line} msg]
    } else {
	set result [catch {eval $line} msg]
    }

    if {[string length $msg] > 0} {
	if {$result == 1} {
	    puts "Error: $msg"
	} else {
	    puts $msg
	}
    }
}
