# include <stdio.h>
# include <time.h>
# include <ctype.h>

# include "hfs.h"
# include "hcwd.h"
# include "hfsutil.h"
# include "hdir.h"

/*
 * NAME:	timestr()
 * DESCRIPTION:	convert seconds to printable string for directory listings
 */
static
char *timestr(long secs)
{
  static char str[18];
  struct tm *t;
  static char *months[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun",
			    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

  /* MMM DD HH:MM YYYY */

  t = localtime(&secs);

  sprintf(str, "%s %2d %02d:%02d %4d",
	  months[t->tm_mon], t->tm_mday,
	  t->tm_hour, t->tm_min, 1900 + t->tm_year);

  return str;
}

/*
 * NAME:	cleanstr()
 * DESCRIPTION:	sanitize a string for printable characters
 */
static
void cleanstr(char *dest, char *str)
{
  for ( ; *str; ++str)
    *dest++ = isprint(*str) ? *str : '?';

  *dest = 0;
}

int hdir_main(int argc, char *argv[])
{
  hfsvol *vol;
  hfsdir *dir;
  hfsdirent ent;
  char *path;

  if (argc > 2)
    {
      fprintf(stderr, "Usage: %s [hfs-path]\n", argv[0]);
      return 1;
    }

  vol = hfs_remount(argv[0], hcwd_getvol(-1));
  if (vol == 0)
    return 1;

  if (argc == 2)
    path = argv[1];
  else
    path = ":";

  dir = hfs_opendir(vol, path);
  if (dir == 0)
    {
      hfs_perror("Can't open HFS directory");
      return 1;
    }

  while (hfs_readdir(dir, &ent) >= 0)
    {
      char name[32];

      if (ent.flags & HFS_ISDIR)
	{
	  cleanstr(name, ent.name);

	  printf("d %9lu item%c               %s %s\n",
		 ent.dsize, ent.dsize == 1 ? ' ' : 's',
		 timestr(ent.mddate), name);
	}
      else
	{
	  char type[5], creator[5];

	  cleanstr(type,    ent.type);
	  cleanstr(creator, ent.creator);
	  cleanstr(name,    ent.name);

	  printf("%c %4s/%4s %9lu %9lu %s %s\n",
		 ent.flags & HFS_ISLOCKED ? 'F' : 'f',
		 type, creator, ent.rsize, ent.dsize,
		 timestr(ent.mddate), name);
	}
    }

  if (hfs_closedir(dir) < 0)
    {
      hfs_perror("Error closing HFS directory");
      return 1;
    }

  if (hfs_umount(vol) < 0)
    {
      hfs_perror("Error closing HFS volume");
      return 1;
    }

  return 0;
}
