#!./hfswish

###############################################################################
#
# hfsutils - tools for reading and writing Macintosh HFS volumes
# Copyright (C) 1996 Robert Leslie
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
###############################################################################

set tk_strictMotif 1

set unique 0

proc unique {} {
    global unique

    return [incr unique]
}

proc makegui {} {
    proc makelist {w click other} {
	frame $w

	menubutton $w.mb -textvariable cwd($w)  \
		-indicatoron 1 -menu $w.mb.m  \
		-relief raised -bd 2 -padx 4p -pady 4p  \
		-highlightthickness 2 -anchor c
	menu $w.mb.m -tearoff 0

	frame $w.mid

	listbox $w.mid.lb -xscroll "$w.bot.sb set" -yscroll "$w.mid.sb set"  \
		-exportselection 0 -width 25 -selectmode extended
	scrollbar $w.mid.sb -orient vert -command "$w.mid.lb yview"

	bind $w.mid.lb <Button-1> [list $other selection clear 0 end]
	bind $w.mid.lb <KeyPress> [list $other selection clear 0 end]
	bind $w.mid.lb <ButtonRelease-1> [list wclick $w $click]
	bind $w.mid.lb <KeyRelease> [list wclick $w $click]
	bind $w.mid.lb <Double-ButtonRelease-1> ".m.1.ops.view invoke"
	bind $w.mid.lb <Key-Return> ".m.1.ops.view invoke"

	pack $w.mid.sb -side right -fill y
	pack $w.mid.lb -side right -fill both -expand 1

	frame $w.bot

	scrollbar $w.bot.sb -orient horiz -command "$w.mid.lb xview"
	frame $w.bot.c -height 24 -width 24

	pack $w.bot.c -side right
	pack $w.bot.sb -side bottom -fill x

	pack $w.mb -side top
	pack $w.bot -side bottom -fill x
	pack $w.mid -side top -fill both -expand 1
    }

    proc wclick {side copytext} {
	set box $side.mid.lb
	set sel [$box curselection]

	global vol xmode dir fstype

	if {[llength $sel] == 0} {
	    .m.1.ops.copy config -state disabled -text "Copy"
	    .m.1.ops.view config -state disabled -text "View"
	    .m.1.ops.info config -state disabled
	} elseif {[llength $sel] == 1} {
	    .m.1.ops.copy config -state normal -text $copytext  \
		    -command [list do_copy $side]
	    .m.1.ops.view config -state normal -command [list do_open $side]
	    .m.1.ops.info config -state normal -command [list do_info $side]

	    set ind [lindex $sel 0]
	    set name [$box get $ind]

	    if {[regexp "[$vol($side) sepchar]\$" $name]} {
		.m.1.ops.view config -text "Open"
	    } else {
		.m.1.ops.view config -text "View"
	    }

	    assoc [lindex $dir($side) $ind] item

	    set vstate disabled
	    switch -glob $item(kind) {
		directory {
		    set vstate normal
		}

		file {
		    switch $fstype($side) {
			hfs {
			    if {[string compare $item(type) "TEXT"] == 0 &&  \
				    $item(dsize) > 0} {
				set vstate normal
			    }
			}
			ufs {
			    set mode [ufs_automode $vol($side) item]
			    if {$mode == "text" || $mode == "binh"} {
				set vstate normal
			    }
			}
		    }
		}
	    }

	    .m.1.ops.view config -state $vstate

	    if {[.m.2.mode.auto cget -state] != "disabled"} {
		if {[info exists mode]} {
		    set xmode $mode
		} else {
		    set xmode [$fstype($side)_automode $vol($side) item]
		}
	    }
	} else {
	    .m.1.ops.copy config -state normal -text $copytext  \
		    -command [list do_copy $side]
	    .m.1.ops.view config -state disabled -text "View"
	    # .m.1.ops.info config -state normal -command [list do_info $side]
	    .m.1.ops.info config -state disabled

	    if {[.m.2.mode.auto cget -state] != "disabled"} {
		foreach ind $sel {
		    assoc [lindex $dir($side) $ind] item
		    set auto [$fstype($side)_automode $vol($side) item]

		    if {[info exists mode]} {
			if {$mode != $auto} {
			    set mode auto
			    break
			}
		    } else {
			set mode $auto
		    }
		}

		set xmode $mode
	    }
	}
    }

    proc do_copy {side} {
	set box $side.mid.lb
	set sel [$box curselection]

	global fstype dir vol xmode

	if {$side == ".l"} {
	    set other ".r"
	} else {
	    set other ".l"
	}

	if {$fstype($other) == "hfs"} {
	    error "can't yet copy to HFS volumes"
	}

	watch_cursor
	after idle "updatelist $other"

	set mode $xmode

	foreach ind $sel {
	    assoc [lindex $dir($side) $ind] item

	    if {$item(kind) == "directory"} {
		error "can't yet copy whole directories"
	    }

	    if {$xmode == "auto"} {
		set mode [$fstype($side)_automode $vol($side) item]
	    }

	    $vol($side) copyout $mode $item(name) [$vol($other) cwd]
	}
    }

    proc do_info {side} {
	set box $side.mid.lb
	set sel [$box curselection]

	global dir

	foreach elt $sel {
	    show_info [lindex $dir($side) $elt]
	}
    }

    proc do_open {side} {
	set box $side.mid.lb
	set sel [$box curselection]
	set ind [lindex $sel 0]

	global vol

	set name [$box get $ind]
	if {[regexp "(.*)[$vol($side) sepchar]\$" $name ignore name]} {
	    mchdir $side $name
	    return
	}

	show_file $vol($side) $name
    }

    makelist .l ">> Copy >>" .r.mid.lb
    makelist .r "<< Copy <<" .l.mid.lb

    rename makelist ""


    # Middle controls

    frame .m

    frame .m.0 -height 24

    frame .m.1 -bd 4
    label .m.1.l -text "Options"
    frame .m.1.ops -bd 2 -relief groove
    button .m.1.ops.copy -state disabled -text "Copy"
    button .m.1.ops.view -state disabled -text "View"
    button .m.1.ops.info -state disabled -text "Get Info"

    pack .m.1.ops.copy .m.1.ops.view .m.1.ops.info  \
	    -side top -pady 1m -padx 1m -fill x
    pack .m.1.l .m.1.ops -side top -fill x

    frame .m.2 -bd 4
    label .m.2.l -text "Mode"
    frame .m.2.mode -bd 2 -relief groove

    radiobutton .m.2.mode.auto -text "Automatic" -var xmode -value auto
    radiobutton .m.2.mode.macb -text "MacBinary II" -var xmode -value macb
    radiobutton .m.2.mode.binh -text "BinHex" -var xmode -value binh
    radiobutton .m.2.mode.text -text "Text" -var xmode -value text
    radiobutton .m.2.mode.raw -text "Raw Data" -var xmode -value raw

    global xmode
    set xmode auto

    pack .m.2.mode.auto  \
	    .m.2.mode.macb  \
	    .m.2.mode.binh  \
	    .m.2.mode.text  \
	    .m.2.mode.raw  \
	    -side top -anchor w
    pack .m.2.l .m.2.mode -side top -fill x

    frame .m.3 -height 24

    pack .m.0 .m.1 .m.2 .m.3 -side top -fill x


    # Menu bar

    frame .mb -relief raised -bd 2
    menubutton .mb.file -text "File" -menu .mb.file.m
    # menubutton .mb.view -text "View" -menu .mb.view.m
    # menubutton .mb.actn -text "Action" -menu .mb.actn.m

    pack .mb.file -side left

    menu .mb.file.m -tearoff 0
    foreach item {
	{command -label "Show License" -command show_license}
	{separator}
	{command -label "Exit" -command "destroy ."}
    } {eval .mb.file.m add $item}

    # menu .mb.view.m -tearoff 0
    # foreach item {
	# {checkbutton -label "Show All Files"}
    # } {eval .mb.view.m add $item}

    # menu .mb.actn.m
    # foreach item {
	# {command -label "Copy File(s)"}
	# {command -label "View File"}
	# {command -label "Get File Info"}
	# {separator}
	# {command -label "Delete File(s)"}
    # } {eval .mb.actn.m add $item}


    # Put it together

    pack .mb -side top -fill x
    pack .l -side left -fill both -expand 1
    pack .r -side right -fill both -expand 1
    pack .m -side left

    wm title . "HFS Utility"
    wm iconname . "HFS Utility"

    . config -cursor left_ptr
}

proc show_license {} {
    set w ".license"

    catch {destroy $w}

    toplevel $w -cursor left_ptr
    wm title $w "License"

    set text "[hfs version] - [hfs copyright]\n\n[hfs license]"
    regsub -all "(\[^\n])\n(\[^\n])" $text {\1 \2} text
    regsub "\n*\$" $text "" text

    message $w.m -text $text  \
	    -font {-*-new century schoolbook-medium-r-*-*-*-140-*-*-p-*-*-*}

    pack $w.m -expand 1
}

proc hfs_automode {vol array} {
    upvar $array item

    if {$item(kind) == "directory"} {
	return auto
    } elseif {[string compare $item(type) "TEXT"] == 0} {
	return text
    } elseif {$item(rsize) == 0} {
	return raw
    } else {
	return macb
    }
}

proc ufs_automode {vol array} {
    upvar $array item

    set name $item(name)

    if {$item(kind) == "directory"} {
	return auto
    } elseif {[regexp {\.bin$} $name]} {
	return macb
    } elseif {[regexp {\.hqx$} $name]} {
	return binh
    } elseif {[regexp {\.(txt|c|h)$} $name]} {
	return text
    } elseif {[regexp {\.(sit|sea|cpt|tar|gz|Z)$} $name]} {
	return raw
    } else {
	cd [$vol cwd]
	if {[catch {exec file -L $name} type] == 0 &&  \
		[regexp {text} $type]} {
	    return text
	}

	return raw
    }
} 

proc assoc {list var} {
    upvar $var array

    foreach elt $list {
	if {[info exists key]} {
	    set array($key) $elt
	    unset key
	} else {
	    set key $elt
	}
    }
}

proc watch_cursor {{w "."} {default "left_ptr"}} {
    $w config -cursor watch
    update idletasks

    after idle "$w config -cursor $default"
}

proc updatelist {side} {
    set box $side.mid.lb
    $box delete 0 end

    global dir vol
    set dir($side) [$vol($side) dir]

    set sepchar [$vol($side) sepchar]

    foreach ent $dir($side) {
	assoc $ent item

	if {$item(kind) == "directory"} {
	    set name "$item(name)$sepchar"
	} else {
	    set name $item(name)
	}

	$box insert end $name
    }

    set m $side.mb.m
    $m delete 0 end

    set path [$vol($side) path]
    set last 0

    foreach elt $path {
	set partial [$vol($side) abspath [lrange $path 0 $last]]
	incr last

	$m insert 0 command -label $elt -command [list mchdir $side $partial]
    }

#    global dev
#    $m insert 0 separator
#    $m insert 0 command -label $dev($side) -state disabled

    $m insert end separator
    $m insert end command -label "Open Other..." -command [list mopen $side]

    global cwd
    set cwd($side) [lindex $path [expr [llength $path] - 1]]

    .l.mid.lb selection clear 0 end
    .r.mid.lb selection clear 0 end

    .m.1.ops.copy config -state disabled -text "Copy"
    .m.1.ops.view config -state disabled -text "View"
    .m.1.ops.info config -state disabled
}

proc mchdir {side path} {
    global vol

    watch_cursor

    $vol($side) chdir $path
    updatelist $side
}

proc mopen {side} {
    set w .open

    toplevel $w -cursor left_ptr
    wm title $w "Open Other"
    wm iconname $w "Open..."

    frame $w.top -relief raised -bd 1
    pack $w.top -side top -expand 1 -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    label $w.top.lbl -text "UNIX directory or HFS device: "  \
	    -font -*-times-medium-r-normal-*-*-180-*-*-p-*-*-*
    entry $w.top.entry -font -*-courier-medium-r-normal-*-*-160-*-*-m-*-*-*

    pack $w.top.lbl -side left -expand 1 -padx 3m -pady 3m
    pack $w.top.entry -side left -expand 1 -padx 3m -pady 3m

    button $w.bot.cancel -width 8 -text "Cancel" -command [list destroy $w]
    button $w.bot.ok -width 8 -text "OK" -command "mopen2 $w $side"

    bind $w.top.entry <Key-Return> "mopen2 $w $side"

    frame $w.bot.default -relief sunken -bd 1
    raise $w.bot.ok
    pack $w.bot.cancel -side left -expand 1  \
	    -padx 3m -pady 3m -ipadx 2m -ipady 1m
    pack $w.bot.default -side left -expand 1 -padx 3m -pady 2m
    pack $w.bot.ok -in $w.bot.default -side left  \
	    -padx 2m -pady 2m -ipadx 2m -ipady 1m

    set oldfocus [focus]
    grab set $w
    focus $w.top.entry
}

proc mopen2 {w side} {
    $w config -cursor watch

    watch_cursor

    set device [$w.top.entry get]
    destroy $w

    if {[string length $device] > 0} {
	mountdev $side $device
    }
}

proc plural {count single plural} {
    if {$count == 1} {
	return $single
    } else {
	return $plural
    }
}

proc show_file {vol fname} {
    set w ".show[unique]"

    watch_cursor

    # Make sure we can open the file before building the interface
    set fh [$vol open $fname]

    toplevel $w -cursor left_ptr
    wm title $w $fname
    frame $w.l
    text $w.l.text -yscroll "$w.r.yscroll set"  \
	    -height 30 -width 80 -wrap word  \
	    -font -*-lucidatypewriter-medium-r-*-*-*-120-*-*-m-*-*-*
    # -xscroll "$w.l.xscroll set"
    # scrollbar $w.l.xscroll -orient horiz -command "$w.l.text xview"

    frame $w.r
    scrollbar $w.r.yscroll -orient vert -command "$w.l.text yview"
    # frame $w.r.c -height 24 -width 24

    # pack $w.r.c -side bottom
    pack $w.r.yscroll -side right -fill y

    # pack $w.l.xscroll -side bottom -fill x
    pack $w.l.text -side top -fill both -expand 1

    pack $w.r -side right -fill both
    pack $w.l -side right -fill both -expand 1

    watch_cursor $w.l.text xterm

    while {1} {
	set buf [$fh read 512]
	if {[string length $buf] == 0} {
	    $fh close
	    break
	}

	regsub -all "\r\n?" $buf "\n" buf

	$w.l.text insert end $buf
    }

    $w.l.text config -state disabled
}

proc show_info {list} {
    assoc $list info

    set w ".info[unique]"

    toplevel $w -cursor left_ptr
    wm title $w "$info(name) Info"
    wm iconname $w "Info"

    frame $w.left
    frame $w.right

    if {$info(kind) == "directory"} {
	label $w.left.name -text "Name: "
	label $w.left.kind -text "Kind: "
	label $w.left.dsiz -text "Contains: "
	label $w.left.mdat -text "Last Modified: "

	set size "$info(size) [plural $info(size) item items]"

	label $w.right.name -text $info(name)
	label $w.right.kind -text $info(kind)
	label $w.right.dsiz -text $size
	label $w.right.mdat -text [ctime $info(mddate)]

	pack $w.left.name $w.left.kind $w.left.dsiz $w.left.mdat  \
	    -side top -anchor e -fill y -expand 1
	pack $w.right.name $w.right.kind $w.right.dsiz $w.right.mdat  \
	    -side top -anchor w -fill y -expand 1
    } else {
	label $w.left.name -text "Name: "
	label $w.left.kind -text "Kind: "

	label $w.right.name -text $info(name)
	label $w.right.kind -text $info(kind)

	if {[info exists info(type)]} {
	    label $w.left.type -text "Type: "
	    label $w.left.crea -text "Creator: "
	    label $w.left.rsiz -text "Resource Fork: "
	    label $w.left.dsiz -text "Data Fork: "

	    set rsiz "$info(rsize) [plural $info(rsize) byte bytes]"
	    set dsiz "$info(dsize) [plural $info(dsize) byte bytes]"

	    label $w.right.type -text $info(type)  \
		    -font -*-lucidatypewriter-medium-r-*-*-*-120-*-*-m-*-*-*
	    label $w.right.crea -text $info(creator)  \
		    -font -*-lucidatypewriter-medium-r-*-*-*-120-*-*-m-*-*-*
	    label $w.right.rsiz -text $rsiz
	    label $w.right.dsiz -text $dsiz
	} else {
	    label $w.left.size -text "Size: "

	    set size "$info(size) [plural $info(size) byte bytes]"

	    label $w.right.size -text $size
	}

	label $w.left.mdat -text "Last Modified: "

	label $w.right.mdat -text [ctime $info(mddate)]

	if {[info exists info(type)]} {
	    pack $w.left.name $w.left.kind $w.left.type $w.left.crea  \
		    $w.left.rsiz $w.left.dsiz $w.left.mdat  \
		    -side top -anchor e -fill y -expand 1
	    pack $w.right.name $w.right.kind $w.right.type $w.right.crea  \
		    $w.right.rsiz $w.right.dsiz $w.right.mdat  \
		    -side top -anchor w -fill y -expand 1
	} else {
	    pack $w.left.name $w.left.kind $w.left.size $w.left.mdat  \
		    -side top -anchor e -fill y -expand 1
	    pack $w.right.name $w.right.kind $w.right.size $w.right.mdat  \
		    -side top -anchor w -fill y -expand 1
	}
    }

    pack $w.left $w.right -side left -fill x -expand 1
}

proc ufs_handle {dir} {
    set handle "ufsvol[unique]"

    global ufsdir

    cd $dir
    set ufsdir($handle) [pwd]

    proc $handle {verb args} {
	set handle [lindex [info level [info level]] 0]
	global ufsdir

	switch -glob $verb {
	    umount {
		unset ufsdir($handle)
		rename $handle ""
	    }

	    path {
		set pwd $ufsdir($handle)
		set path [split $pwd "/"]

		if {[string compare [lindex $path 0] ""] == 0} {
		    set path [lreplace $path 0 0 "/"]
		}

		if {[llength $path] == 2 &&  \
			[string length [lindex $path 1]] == 0} {
		    set path [lreplace $path 1 1]
		}

		return $path
	    }

	    abspath {
		set path [lindex $args 0]

		if {[string compare [lindex $path 0] "/"] == 0} {
		    set path [lreplace $path 0 0 ""]
		}

		if {[llength $path] == 1 &&  \
			[string length [lindex $path 0]] == 0} {
		    return "/"
		} else {
		    return [join $path "/"]
		}
	    }

	    dir {
		set pwd $ufsdir($handle)
		set list [list]

		cd $pwd
		set files [lsort [glob .* *]]

		foreach name $files {
		    if {[string compare $name "."] == 0 ||  \
			    [string compare $name ".."] == 0} {
			continue
		    } else {
			if {[catch {file stat $name stat}]} {
			    continue
			}

			set kind [file type $name]
			set mddate $stat(mtime)

			if {[file isdirectory $name]} {
			    set kind "directory"
			    set size "?"
			    catch {
				set size [expr [llength  \
					[glob $name/.* $name/*]] - 2]
			    }
			} elseif {$kind == "file"} {
			    set size $stat(size)
			} else {
			    continue
			}

			lappend list [list  \
				name $name  \
				kind $kind  \
				size $size  \
				mddate $mddate]
		    }
		}

		return $list
	    }

	    chdir {
		set path [lindex $args 0]
		set pwd $ufsdir($handle)

		if {[string index $pwd 0] != "/"} {
		    cd $pwd
		}
		cd $path

		set ufsdir($handle) [pwd]
	    }

	    cwd {
		return $ufsdir($handle)
	    }

	    sepchar {
		return "/"
	    }

	    open {
		set path [lindex $args 0]
		set pwd $ufsdir($handle)

		cd $pwd
		set fh [open $path]

		proc $fh {verb args} {
		    set fh [lindex [info level [info level]] 0]

		    switch -glob $verb {
			read {
			    set len [lindex $args 0]

			    return [read $fh $len]
			}

			close {
			    close $fh
			    rename $fh ""
			}

			* {
			    error "unknown call to $fh $verb $args"
			}
		    }
		}

		return $fh
	    }

	    copyout {
		set mode [lindex $args 0]
		set path [lindex $args 1]
		set dest [lindex $args 2]

		set pwd $ufsdir($handle)

		cd $pwd
		exec cp -f $path $dest
	    }

	    * {
		error "unknown call to $handle $verb $args"
	    }
	}
    }

    return $handle
}

proc ufs {verb args} {
    switch $verb {
	mount {
	    set dir [lindex $args 0]

	    return [ufs_handle $dir]
	}

	* {
	    error "bad arg 1 to ufs"
	}
    }
}

proc umountdev {side} {
    global vol

    if {[info exists vol($side)]} {
	$vol($side) umount
	unset vol($side)
    }
}

proc mountdev {side device} {
    global vol dev appcwd fstype xmode

    cd $appcwd

    if {! [file exists $device]} {
	error "can't open $device (no such file or directory)"
    }

    if {[file isdirectory $device]} {
	set handle [ufs mount $device]
	set fstype($side) ufs
    } else {
	set handle [hfs mount $device]
	set fstype($side) hfs
    }

    umountdev $side
    set dev($side) $device

    set vol($side) $handle
    updatelist $side

    if {$fstype(.l) == $fstype(.r)} {
	set state disabled
	set xmode auto
    } else {
	set state normal
    }

    foreach mode {auto macb binh text raw} {
	.m.2.mode.$mode config -state $state
    }
}

###############################################################################

if {[string compare [lindex $argv 0] "--license"] == 0} {
    puts -nonewline "\n[hfs license]"
    exit
}

if {[string compare [lindex $argv 0] "--version"] == 0} {
    puts "[hfs version] - [hfs copyright]"
    puts "`$argv0 --license' for licensing information."
    exit
}

###############################################################################

rename exit _exit

proc exit {{code 0}} {
    umountdev .l
    umountdev .r

    _exit $code
}

###############################################################################

tk appname xhfs

makegui

set appcwd [pwd]

set fstype(.l) none
set fstype(.r) none

if {$argc > 2} {
    puts "Usage: $argv0 \[left-vol \[right-vol]]"
    exit 1
}

if {$argc > 0} {
    mountdev .l [lindex $argv 0]
    if {$argc > 1} {
	mountdev .r [lindex $argv 1]
    } else {
	mountdev .r $appcwd
    }
} else {
    mountdev .l $appcwd
    mountdev .r $appcwd
}
