# include <string.h>
# include <stdlib.h>
# include <stdio.h>
# include <time.h>
# include <unistd.h>
# include <sys/stat.h>
# include <fcntl.h>
# include <tcl.h>

# ifdef DMALLOC
# include <dmalloc.h>
# endif

# include "tclhfs.h"
# include "hfs.h"
# include "copyout.h"
# include "version.h"

static int umount_error, close_error;

/*
 * NAME:	Hfs_DirentStr()
 * DESCRIPTION:	return a Tcl stat list for a single HFS entity
 */
static
char *Hfs_DirentStr(hfsdirent *ent)
{
  char cnid[12], rsize[12], dsize[12], crdate[12], mddate[12];
  char *argv[18];
  register int argc;

  sprintf(cnid,   "%lu", ent->cnid);
  sprintf(rsize,  "%lu", ent->rsize);
  sprintf(dsize,  "%lu", ent->dsize);
  sprintf(crdate, "%lu", ent->crdate);
  sprintf(mddate, "%lu", ent->mddate);

  argc = 0;

  argv[argc++] = "name";
  argv[argc++] = ent->name;

  argv[argc++] = "kind";

  if (ent->flags & HFS_ISDIR)
    {
      argv[argc++] = "directory";

      argv[argc++] = "dirid";
      argv[argc++] = cnid;

      argv[argc++] = "size";
      argv[argc++] = dsize;

      argv[argc++] = "crdate";
      argv[argc++] = crdate;

      argv[argc++] = "mddate";
      argv[argc++] = mddate;
    }
  else
    {
      argv[argc++] = "file";

      argv[argc++] = "fileid";
      argv[argc++] = cnid;

      argv[argc++] = "type";
      argv[argc++] = ent->type;

      argv[argc++] = "creator";
      argv[argc++] = ent->creator;

      argv[argc++] = "rsize";
      argv[argc++] = rsize;

      argv[argc++] = "dsize";
      argv[argc++] = dsize;

      argv[argc++] = "crdate";
      argv[argc++] = crdate;

      argv[argc++] = "mddate";
      argv[argc++] = mddate;
    }

  return Tcl_Merge(argc, argv);
}

/*
 * NAME:	Hfs_GetDir()
 * DESCRIPTION:	collect and return the contents of an HFS directory
 */
static
int Hfs_GetDir(Tcl_Interp *interp, hfsvol *vol, char *path)
{
  hfsdir *dir;
  hfsdirent ent;
  char *str;

  if (hfs_stat(vol, path, &ent) < 0)
    {
      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
      return TCL_ERROR;
    }

  if (ent.flags & HFS_ISDIR)
    {
      dir = hfs_opendir(vol, path);
      if (dir == 0)
	{
	  Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	  return TCL_ERROR;
	}

      while (hfs_readdir(dir, &ent) >= 0)
	{
	  str = Hfs_DirentStr(&ent);
	  if (str == 0)
	    {
	      hfs_closedir(dir);
	      Tcl_SetResult(interp, "out of memory", TCL_STATIC);
	      return TCL_ERROR;
	    }

	  Tcl_AppendElement(interp, str);

	  free(str);
	}

      if (hfs_closedir(dir) < 0)
	{
	  Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	  return TCL_ERROR;
	}
    }
  else  /* ! HFS_ISDIR */
    {
      str = Hfs_DirentStr(&ent);
      if (str == 0)
	{
	  interp->result = "out of memory";
	  return TCL_ERROR;
	}

      Tcl_AppendElement(interp, str);

      free(str);
    }

  return TCL_OK;
}

/*
 * NAME:	Hfs_FileCmd()
 * DESCRIPTION:	operate on an HFS file
 */
static
int Hfs_FileCmd(ClientData clientData, Tcl_Interp *interp,
		int argc, char *argv[])
{
  hfsfile *file = clientData;

  switch (argc)
    {
    case 1:
      interp->result = "missing command";
      return TCL_ERROR;

    case 2:
      if (strcmp(argv[1], "close") == 0)
	{
	  Tcl_DeleteCommand(interp, argv[0]);
	  if (close_error < 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }
	}
      else if (strcmp(argv[1], "tell") == 0)
	{
	  long offset;

	  offset = hfs_lseek(file, 0, SEEK_CUR);
	  if (offset < 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }

	  sprintf(interp->result, "%lu", offset);
	}
      else if (strcmp(argv[1], "stat") == 0)
	{
	  hfsdirent ent;
	  char *str;

	  if (hfs_fstat(file, &ent) < 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }

	  str = Hfs_DirentStr(&ent);
	  if (str == 0)
	    {
	      interp->result = "out of memory";
	      return TCL_ERROR;
	    }

	  Tcl_SetResult(interp, str, TCL_DYNAMIC);
	}
      else
	{
	  Tcl_AppendResult(interp, "bad command \"", argv[1],
			   "\" or wrong # args", (char *) 0);
	  return TCL_ERROR;
	}
      break;

    case 3:
      if (strcmp(argv[1], "fork") == 0)
	{
	  int fork;

	  if (strcmp(argv[2], "data") == 0)
	    fork = 0;
	  else if (strcmp(argv[2], "rsrc") == 0 ||
		   strcmp(argv[2], "resource") == 0)
	    fork = 1;
	  else
	    {
	      interp->result = "bad arg to fork: must be data or rsrc";
	      return TCL_ERROR;
	    }

	  hfs_fork(file, fork);
	}
      else if (strcmp(argv[1], "seek") == 0)
	{
	  long offset;

	  if (Tcl_ExprLong(interp, argv[2], &offset) != TCL_OK)
	    return TCL_ERROR;

	  offset = hfs_lseek(file, offset, SEEK_SET);
	  if (offset < 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }

	  sprintf(interp->result, "%lu", offset);
	}
      else if (strcmp(argv[1], "read") == 0)
	{
	  long bytes;
	  char *mem;

	  if (Tcl_ExprLong(interp, argv[2], &bytes) != TCL_OK)
	    return TCL_ERROR;

	  if (bytes <= 0)
	    {
	      interp->result = "size must be > 0";
	      return TCL_ERROR;
	    }

	  mem = malloc(bytes + 1);
	  if (mem == 0)
	    {
	      interp->result = "out of memory";
	      return TCL_ERROR;
	    }

	  bytes = hfs_read(file, mem, bytes);
	  if (bytes < 0)
	    {
	      free(mem);
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }

	  mem[bytes] = 0;
	  Tcl_SetResult(interp, mem, TCL_DYNAMIC);
	}
      else
	{
	  Tcl_AppendResult(interp, "bad command \"", argv[1],
			   "\" or wrong # args", (char *) 0);
	  return TCL_ERROR;
	}
      break;

    case 4:
      if (strcmp(argv[1], "seek") == 0)
	{
	  long offset;
	  int whence;

	  if (Tcl_ExprLong(interp, argv[2], &offset) != TCL_OK)
	    return TCL_ERROR;

	  if (strcmp(argv[3], "start") == 0 ||
	      strcmp(argv[3], "set") == 0)
	    whence = SEEK_SET;
	  else if (strcmp(argv[3], "current") == 0 ||
		   strcmp(argv[3], "cur") == 0)
	    whence = SEEK_CUR;
	  else if (strcmp(argv[3], "end") == 0)
	    whence = SEEK_END;
	  else
	    {
	      interp->result = "bad arg 3: must be start, current, or end";
	      return TCL_ERROR;
	    }

	  offset = hfs_lseek(file, offset, whence);
	  if (offset < 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }

	  sprintf(interp->result, "%lu", offset);
	}
      else
	{
	  Tcl_AppendResult(interp, "bad command \"", argv[1],
			   "\" or wrong # args", (char *) 0);
	  return TCL_ERROR;
	}
      break;

    default:
      Tcl_AppendResult(interp, "bad command \"", argv[1],
		       "\" or wrong # args", (char *) 0);
      return TCL_ERROR;
    }

  return TCL_OK;
}

/*
 * NAME:	Hfs_DelFile()
 * DESCRIPTION:	called by Tcl when a file reference is deleted
 */
static
void Hfs_DelFile(ClientData clientData)
{
  hfsfile *file = clientData;

  close_error = hfs_close(file);
}

/*
 * NAME:	Hfs_OpenFile()
 * DESCRIPTION:	open an HFS file
 */
static
int Hfs_OpenFile(Tcl_Interp *interp, hfsvol *vol, char *path)
{
  static int id = 0;
  hfsfile *file;

  file = hfs_open(vol, path);
  if (file == 0)
    {
      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
      return TCL_ERROR;
    }

  sprintf(interp->result, "hfsfile%d", id++);

  Tcl_CreateCommand(interp, interp->result, Hfs_FileCmd, file, Hfs_DelFile);

  return TCL_OK;
}

# if 0
/*
 * NAME:	Hfs_Unix2HfsName()
 * DESCRIPTION:	translate a UNIX filename to HFS
 */
static
char *Hfs_Unix2HfsName(char *uname)
{
  static char name[32], *ptr;

  strncpy(name, uname, 31);
  name[31] = 0;

  for (ptr = name; *ptr; ++ptr)
    {
      switch (*ptr)
	{
	case ':':
	  *ptr = '-';
	  break;

	case '_':
	  *ptr = ' ';
	  break;
	}
    }

  return name;
}
# endif

/*
 * NAME:	Hfs_Hfs2UnixName()
 * DESCRIPTION:	translate an HFS filename to UNIX
 */
static
char *Hfs_Hfs2UnixName(char *hfsname, char *ext)
{
  static char name[256], *ptr;

  strcpy(name, hfsname);
  strcat(name, ext);

  for (ptr = name; *ptr; ++ptr)
    {
      switch (*ptr)
	{
	case '/':
	  *ptr = '-';
	  break;

	case ' ':
	  *ptr = '_';
	  break;
	}
    }

  return name;
}

/*
 * NAME:	Hfs_UnixOpen()
 * DESCRIPTION:	open a destination UNIX file
 */
static
int Hfs_UnixOpenW(char *dest, char *name, char *ext)
{
  struct stat sbuf;
  char *path;
  int fd;

  if (stat(dest, &sbuf) == 0 &&
      S_ISDIR(sbuf.st_mode))
    {
      char *uname;

      uname = Hfs_Hfs2UnixName(name, ext);

      path = malloc(strlen(dest) + 1 + strlen(uname) + 1);
      if (path == 0)
	return -1;

      strcpy(path, dest);
      strcat(path, "/");
      strcat(path, uname);
    }
  else
    path = dest;

  fd = open(path, O_WRONLY | O_CREAT | O_TRUNC, 0666);

  if (path != dest)
    free(path);

  if (fd < 0)
    return -1;

  return fd;
}

/*
 * NAME:	Hfs_DelVol()
 * DESCRIPTION:	called by Tcl when a volume reference is deleted
 */
static
void Hfs_DelVol(ClientData clientData)
{
  hfsvol *vol = clientData;

  umount_error = hfs_umount(vol);
}

/*
 * NAME:	Hfs_VolCmd()
 * DESCRIPTION:	operate on an HFS volume
 */
static
int Hfs_VolCmd(ClientData clientData, Tcl_Interp *interp,
	       int argc, char *argv[])
{
  hfsvol *vol = clientData;

  switch (argc)
    {
    case 1:
      interp->result = "missing command";
      return TCL_ERROR;

    case 2:
      if (strcmp(argv[1], "vname") == 0)
	Tcl_SetResult(interp, hfs_vname(vol), TCL_STATIC);
      else if (strcmp(argv[1], "vcrdate") == 0)
	sprintf(interp->result, "%lu", hfs_vcrdate(vol));
      else if (strcmp(argv[1], "vmddate") == 0)
	sprintf(interp->result, "%lu", hfs_vmddate(vol));
      else if (strcmp(argv[1], "freebytes") == 0)
	sprintf(interp->result, "%lu", hfs_freebytes(vol));
      else if (strcmp(argv[1], "umount") == 0)
	{
	  Tcl_DeleteCommand(interp, argv[0]);
	  if (umount_error < 0)
	    {
	      Tcl_CreateCommand(interp, argv[0], Hfs_VolCmd, vol, Hfs_DelVol);

	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }
	}
# if 0
      else if (strcmp(argv[1], "cwd") == 0 ||
	       strcmp(argv[1], "pwd") == 0)
	{
	  char *cwd;

	  cwd = hfs_getcwd(vol);
	  if (cwd == 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }

	  Tcl_SetResult(interp, cwd, TCL_STATIC);
	}
# else
      else if (strcmp(argv[1], "cwdid") == 0)
	{
	  sprintf(interp->result, "%lu", hfs_cwdid(vol));
	}
# endif
      else if (strcmp(argv[1], "path") == 0)
	{
	  char name[32];
	  long id;
	  int listc, i;
	  char **listv;
	  char *result;

	  id = hfs_cwdid(vol);
	  while (id != HFS_CNID_ROOTPAR)
	    {
	      if (hfs_dirinfo(vol, &id, name) < 0)
		{
		  Tcl_SetResult(interp, hfs_error, TCL_STATIC);
		  return TCL_ERROR;
		}

	      Tcl_AppendElement(interp, name);
	    }

	  /* reverse the resulting list */

	  if (Tcl_SplitList(interp, interp->result, &listc, &listv) != TCL_OK)
	    return TCL_ERROR;

	  for (i = 0; i < listc / 2; ++i)
	    {
	      char *tmp;

	      tmp = listv[i];
	      listv[i] = listv[listc - 1 - i];
	      listv[listc - 1 - i] = tmp;
	    }

	  result = Tcl_Merge(listc, listv);
	  free(listv);

	  Tcl_SetResult(interp, result, TCL_DYNAMIC);
	}
      else if (strcmp(argv[1], "dir") == 0)
	{
	  if (Hfs_GetDir(interp, vol, ":") != TCL_OK)
	    return TCL_ERROR;
	}
      else if (strcmp(argv[1], "sepchar") == 0)
	{
	  interp->result = ":";
	}
      else
	{
	  Tcl_AppendResult(interp, "bad command \"", argv[1],
			   "\" or wrong # args", (char *) 0);
	  return TCL_ERROR;
	}
      break;

    case 3:
      if (strcmp(argv[1], "abspath") == 0)
	{
	  int listc, i;
	  char **listv;

	  if (Tcl_SplitList(interp, argv[2], &listc, &listv) != TCL_OK)
	    return TCL_ERROR;

	  for (i = 0; i < listc; ++i)
	    {
	      if (i == 0)
		Tcl_AppendResult(interp, listv[i], (char *) 0);
	      else
		Tcl_AppendResult(interp, ":", listv[i], (char *) 0);
	    }

	  if (listc == 1)
	    Tcl_AppendResult(interp, ":", (char *) 0);

	  free(listv);
	}
      else if (strcmp(argv[1], "cd") == 0 ||
	  strcmp(argv[1], "chdir") == 0)
	{
	  if (hfs_chdir(vol, argv[2]) < 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }
	}
      else if (strcmp(argv[1], "dirinfo") == 0)
	{
	  long id;
	  char name[32], idstr[12];

	  if (Tcl_ExprLong(interp, argv[2], &id) != TCL_OK)
	    return TCL_ERROR;

	  if (hfs_dirinfo(vol, &id, name) < 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }

	  sprintf(idstr, "%lu", id);
	  Tcl_AppendElement(interp, name);
	  Tcl_AppendElement(interp, idstr);
	}
      else if (strcmp(argv[1], "dir") == 0)
	{
	  if (Hfs_GetDir(interp, vol, argv[2]) != TCL_OK)
	    return TCL_ERROR;
	}
      else if (strcmp(argv[1], "open") == 0)
	{
	  if (Hfs_OpenFile(interp, vol, argv[2]) != TCL_OK)
	    return TCL_ERROR;
	}
      else if (strcmp(argv[1], "stat") == 0)
	{
	  hfsdirent ent;
	  char *str;

	  if (hfs_stat(vol, argv[2], &ent) < 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }

	  str = Hfs_DirentStr(&ent);
	  if (str == 0)
	    {
	      interp->result = "out of memory";
	      return TCL_ERROR;
	    }

	  Tcl_SetResult(interp, str, TCL_DYNAMIC);
	}
      else if (strcmp(argv[1], "mkdir") == 0)
	{
	  if (hfs_mkdir(vol, argv[2]) < 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }
	}
      else if (strcmp(argv[1], "rmdir") == 0)
	{
	  if (hfs_rmdir(vol, argv[2]) < 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }
	}
      else if (strcmp(argv[1], "delete") == 0)
	{
	  if (hfs_delete(vol, argv[2]) < 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }
	}
      else
	{
	  Tcl_AppendResult(interp, "bad command \"", argv[1],
			   "\" or wrong # args", (char *) 0);
	  return TCL_ERROR;
	}
      break;

    case 4:
      if (strcmp(argv[1], "rename") == 0)
	{
	  if (hfs_rename(vol, argv[2], argv[3]) < 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }
	}
      else if (strcmp(argv[1], "move") == 0)
	{
	  if (hfs_move(vol, argv[2], argv[3]) < 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }
	}
      else
	{
	  Tcl_AppendResult(interp, "bad command \"", argv[1],
			   "\" or wrong # args", (char *) 0);
	  return TCL_ERROR;
	}
      break;

    case 5:
      if (strcmp(argv[1], "create") == 0)
	{
	  if (hfs_create(vol, argv[2], argv[3], argv[4]) < 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }
	}
      else if (strcmp(argv[1], "copyout") == 0)
	{
	  int (*copyfile)(Tcl_Interp *, hfsfile *, int);
	  hfsfile *ifile;
	  int ofile;
	  char *ext;

	  if (strcmp(argv[2], "macbinary") == 0 ||
	      strcmp(argv[2], "macb") == 0)
	    {
	      copyfile = Hfs_CopyOutMacb;
	      ext = ".bin";
	    }
	  else if (strcmp(argv[2], "binhex") == 0 ||
		   strcmp(argv[2], "binh") == 0)
	    {
	      copyfile = Hfs_CopyOutBinh;
	      ext = ".hqx";
	    }
	  else if (strcmp(argv[2], "text") == 0)
	    {
	      copyfile = Hfs_CopyOutText;
	      ext = ".txt";
	    }
	  else if (strcmp(argv[2], "raw") == 0 ||
		   strcmp(argv[2], "data") == 0)
	    {
	      copyfile = Hfs_CopyOutRaw;
	      ext = "";
	    }
	  else
	    {
	      interp->result = "bad mode: must be macb, binh, text, or raw";
	      return TCL_ERROR;
	    }

	  copyout_dest_name = argv[3];

	  ifile = hfs_open(vol, copyout_dest_name);
	  if (ifile == 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }

	  ofile = Hfs_UnixOpenW(argv[4], copyout_dest_name, ext);
	  if (ofile < 0)
	    {
	      interp->result = "can't open destination for writing";
	      hfs_close(ifile);
	      return TCL_ERROR;
	    }

	  if (copyfile(interp, ifile, ofile) != TCL_OK)
	    {
	      /* delete file? path not saved... */

	      close(ofile);
	      hfs_close(ifile);
	      return TCL_ERROR;
	    }

	  if (close(ofile) < 0)
	    {
	      interp->result = "failed to close destination file";
	      hfs_close(ifile);
	      return TCL_ERROR;
	    }

	  if (hfs_close(ifile) < 0)
	    {
	      Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	      return TCL_ERROR;
	    }
	}
      else
	{
	  Tcl_AppendResult(interp, "bad command \"", argv[1],
			   "\" or wrong # args", (char *) 0);
	  return TCL_ERROR;
	}
      break;

    default:
      Tcl_AppendResult(interp, "bad command \"", argv[1],
		       "\" or wrong # args", (char *) 0);
      return TCL_ERROR;
    }

  return TCL_OK;
}

/*
 * NAME:	Hfs_Cmd()
 * DESCRIPTION:	Tcl HFS command callback
 */
static
int Hfs_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
  static int id = 0;

  if (argc < 2)
    {
      interp->result = "wrong # args";
      return TCL_ERROR;
    }

  if (strcmp(argv[1], "mount") == 0)
    {
      int partno = 1;
      hfsvol *vol;

      if (argc < 3 || argc > 4)
	{
	  interp->result = "wrong # args";
	  return TCL_ERROR;
	}

      if (argc == 4 &&
	  Tcl_GetInt(interp, argv[3], &partno) != TCL_OK)
	return TCL_ERROR;

      vol = hfs_mount(argv[2], partno);
      if (vol == 0)
	{
	  Tcl_AppendResult(interp, "can't mount ", argv[2],
			   ": ", hfs_error, (char *) 0);
	  return TCL_ERROR;
	}

      sprintf(interp->result, "hfsvol%d", id++);

      Tcl_CreateCommand(interp, interp->result, Hfs_VolCmd, vol, Hfs_DelVol);
    }
  else if (strcmp(argv[1], "format") == 0)
    {
      int partno = 0;

      if (argc != 5)
	{
	  interp->result = "wrong # args";
	  return TCL_ERROR;
	}

      if (Tcl_GetInt(interp, argv[3], &partno) != TCL_OK)
	return TCL_ERROR;

      if (hfs_format(argv[2], partno, argv[4]) < 0)
	{
	  Tcl_SetResult(interp, hfs_error, TCL_STATIC);
	  return TCL_ERROR;
	}
    }
  else if (strcmp(argv[1], "version") == 0)
    {
      if (argc != 2)
	{
	  interp->result = "wrong # args";
	  return TCL_ERROR;
	}

      Tcl_SetResult(interp, version, TCL_STATIC);
    }
  else if (strcmp(argv[1], "copyright") == 0)
    {
      if (argc != 2)
	{
	  interp->result = "wrong # args";
	  return TCL_ERROR;
	}

      Tcl_SetResult(interp, copyright, TCL_STATIC);
    }
  else if (strcmp(argv[1], "license") == 0)
    {
      if (argc != 2)
	{
	  interp->result = "wrong # args";
	  return TCL_ERROR;
	}

      Tcl_SetResult(interp, license, TCL_STATIC);
    }
  else
    {
      Tcl_AppendResult(interp, "bad hfs command \"", argv[1],
		       "\": should be mount", (char *) 0);
      return TCL_ERROR;
    }

  return TCL_OK;
}

/*
 * NAME:	Hfs_CtimeCmd()
 * DESCRIPTION:	implementation for ctime Tcl command
 */
static
int Hfs_CtimeCmd(ClientData clientData, Tcl_Interp *interp,
		 int argc, char *argv[])
{
  long date;
  static char str[25];

  if (argc != 2)
    {
      interp->result = "wrong # args";
      return TCL_ERROR;
    }

  if (Tcl_ExprLong(interp, argv[1], &date) != TCL_OK)
    return TCL_ERROR;

  strcpy(str, ctime(&date));
  str[24] = 0;

  interp->result = str;

  return TCL_OK;
}

/*
 * NAME:	Hfs_Init()
 * DESCRIPTION:	initialize Tcl components for HFS handling
 */
int Hfs_Init(Tcl_Interp *interp)
{
  Tcl_CreateCommand(interp, "hfs", Hfs_Cmd, 0, 0);
  Tcl_CreateCommand(interp, "ctime", Hfs_CtimeCmd, 0, 0);

  return TCL_OK;
}
