# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include <errno.h>
# include <unistd.h>

# ifdef DMALLOC
# include <dmalloc.h>
# endif

# include "hcwd.h"

# define STATEFNAME  ".hcwd"
# define TABCHUNKSZ  8

static FILE *statef = 0;
static mountent *mounts = 0;
static int mtabsz = 0, nmounts = 0;
static int curvol = -1, dirty = 0;

/*
 * NAME:	addent()
 * DESCRIPTION:	insert mount entry into table
 */
static
int addent(mountent *ent)
{
  if (nmounts >= mtabsz)
    {
      mountent *newmem;

      mtabsz += TABCHUNKSZ;
      newmem = mounts ? realloc(mounts, mtabsz * sizeof(mountent)) :
	malloc(mtabsz * sizeof(mountent));
      if (newmem == 0)
	return -1;

      mounts = newmem;
    }

  mounts[nmounts++] = *ent;

  dirty = 1;

  return 0;
}

/*
 * NAME:	hcwd_init()
 * DESCRIPTION:	read state file
 */
int hcwd_init(void)
{
  char *home, *path;
  char buf[512], *start, *ptr;
  mountent entry;
  int newcur = -1;

  home = getenv("HOME");
  if (home == 0)
    home = "";

  path = malloc(strlen(home) + 1 + strlen(STATEFNAME) + 1);
  if (path == 0)
    return -1;

  strcpy(path, home);
  strcat(path, "/" STATEFNAME);

  statef = fopen(path, "r+");
  if (statef == 0 && errno == ENOENT)
    statef = fopen(path, "w+");

  free(path);

  if (statef == 0)
    return -1;

  if (fgets(buf, sizeof(buf), statef))
    newcur = atoi(buf);

  while (fgets(buf, sizeof(buf), statef))
    {
      start = ptr = buf;

      while (*ptr && *ptr != '\n' && *ptr != '\t')
	++ptr;

      if (*ptr == '\t')
	{
	  *ptr++ = 0;
	  entry.vname = strdup(start);
	  if (entry.vname == 0)
	    return -1;

	  start = ptr;
	}

      while (*ptr && *ptr != '\n' && *ptr != '\t')
	++ptr;

      if (*ptr == '\t')
	{
	  *ptr++ = 0;
	  entry.vcrdate = strtol(start, 0, 0);
	  start = ptr;
	}

      while (*ptr && *ptr != '\n' && *ptr != '\t')
	++ptr;

      if (*ptr == '\t')
	{
	  *ptr++ = 0;
	  entry.path = strdup(start);
	  if (entry.path == 0)
	    return -1;

	  start = ptr;
	}

      while (*ptr && *ptr != '\n' && *ptr != '\t')
	++ptr;

      if (*ptr == '\t')
	{
	  *ptr++ = 0;
	  entry.partno = atoi(start);
	  start = ptr;
	}

      while (*ptr && *ptr != '\n' && *ptr != '\t')
	++ptr;

      if (*ptr == '\n' || *ptr == '\t')
	*ptr = 0;

      if (*start)
	{
	  entry.cwd = strdup(start);
	  if (entry.cwd == 0)
	    return -1;

	  if (addent(&entry) < 0)
	    return -1;
	}
    }

  curvol = (newcur >= 0 && newcur < nmounts) ? newcur : -1;

  return 0;
}

/*
 * NAME:	hcwd_finish()
 * DESCRIPTION:	flush changes to state file
 */
int hcwd_finish(void)
{
  if (statef && mounts && dirty)
    {
      int i;

      rewind(statef);
      if (ftruncate(fileno(statef), 0) < 0)
	return -1;

      fprintf(statef, "%d\n", curvol);

      for (i = 0; i < nmounts; ++i)
	fprintf(statef, "%s\t%lu\t%s\t%d\t%s\n",
		mounts[i].vname,
		mounts[i].vcrdate,
		mounts[i].path,
		mounts[i].partno,
		mounts[i].cwd);

      dirty = 0;
    }

  if (mounts)
    {
      mountent *ent;

      for (ent = mounts; ent < mounts + nmounts; ++ent)
	{
	  if (ent->vname)
	    free(ent->vname);
	  if (ent->path)
	    free(ent->path);
	  if (ent->cwd)
	    free(ent->cwd);
	}

      free(mounts);
    }

  if (statef && fclose(statef) < 0)
    return -1;

  return 0;
}

/*
 * NAME:	hcwd_mounted()
 * DESCRIPTION:	register a mounted volume
 */
int hcwd_mounted(char *vname, long vcrdate, char *path, int partno)
{
  mountent *entry, new;

  for (entry = mounts; entry < mounts + nmounts; ++entry)
    {
      if (strcmp(entry->path, path) == 0 &&
	  entry->partno == partno)
	{
	  /* update entry */

	  if (entry->vname)
	    free(entry->vname);
	  if (entry->cwd)
	    free(entry->cwd);

	  entry->vname   = strdup(vname);
	  entry->vcrdate = vcrdate;
	  entry->cwd     = strdup(":");
	  if (entry->vname == 0 ||
	      entry->cwd == 0)
	    return -1;

	  curvol = entry - mounts;
	  dirty  = 1;

	  return 0;
	}
    }

  new.vname   = strdup(vname);
  new.vcrdate = vcrdate;
  new.path    = strdup(path);
  new.partno  = partno;
  new.cwd     = strdup(":");

  if (new.vname == 0 ||
      new.path  == 0 ||
      new.cwd   == 0)
    return -1;

  if (addent(&new) < 0)
    return -1;

  curvol = nmounts - 1;

  return 0;
}

/*
 * NAME:	hcwd_getvol()
 * DESCRIPTION:	return a mount entity
 */
mountent *hcwd_getvol(int vol)
{
  if (vol < 0)
    vol = curvol;

  if (vol < 0 || vol >= nmounts)
    return 0;

  return &mounts[vol];
}

/*
 * NAME:	hcwd_setvol()
 * DESCRIPTION:	change the current volume
 */
int hcwd_setvol(int vol)
{
  if (vol < 0 || vol >= nmounts)
    return -1;

  if (curvol != vol)
    {
      curvol = vol;
      dirty  = 1;
    }

  return 0;
}

/*
 * NAME:	hcwd_setcwd()
 * DESCRIPTION:	change the current working directory for a mount entity
 */
int hcwd_setcwd(mountent *ent, char *newcwd)
{
  char *path, *cwd;

  path = newcwd;
  while (*path && *path != ':')
    ++path;

  cwd = (*path == 0) ? strdup(":") : strdup(path);

  if (cwd == 0)
    return -1;

  if (ent->cwd)
    free(ent->cwd);

  ent->cwd = cwd;
  dirty    = 1;

  return 0;
}
