# include <unistd.h>
# include <netinet/in.h>

# include "binhex.h"
# include "crc.h"

static int file;			/* output file fd */
static char line[65];			/* ASCII line buffer */
static int lptr;			/* next char in line buffer */
static int state86;			/* 8->6 encoding state */
static unsigned char lastch;		/* last encoded data byte */
static int runlen;			/* runlength of last data byte */
static unsigned short crc;		/* incremental CRC word */

/*
 * NAME:	bh_start()
 * DESCRIPTION:	begin BinHex encoding
 */
int bh_start(int fd)
{
  file = fd;

  line[0] = ':';
  lptr = 1;

  state86 = 0;
  runlen  = 0;

  crc = 0x0000;

  if (write(file, "(This file must be converted with BinHex 4.0)\n", 46) < 0)
    return -1;

  return 0;
}

/*
 * NAME:	flushline()
 * DESCRIPTION:	flush a line to the output file
 */
static
int flushline(void)
{
  line[lptr++] = '\n';
  if (write(file, line, lptr) < 0)
    return -1;

  lptr = 0;

  return 0;
}

/*
 * NAME:	addchars()
 * DESCRIPTION:	insert bytes of data to the output stream
 */
static
int addchars(char *data, register int len)
{
  char map[] = "!\"#$%&'()*+,-012345689@ABCDEFGHI"
               "JKLMNPQRSTUVXYZ[`abcdefhijklmpqr";
  register unsigned char c;

  while (len--)
    {
      c = *data++;

      if (lptr == sizeof(line) - 1)
	{
	  if (flushline() < 0)
	    return -1;
	}

      switch (state86 & 0xff00)
	{
	case 0x0000:
	  line[lptr++] = map[c >> 2];
	  state86 = 0x0100 | (c & 0x03);
	  break;

	case 0x0100:
	  line[lptr++] = map[((state86 & 0x03) << 4) | (c >> 4)];
	  state86 = 0x0200 | (c & 0x0f);
	  break;

	case 0x0200:
	  line[lptr++] = map[((state86 & 0x0f) << 2) | (c >> 6)];

	  if (lptr == sizeof(line) - 1)
	    {
	      if (flushline() < 0)
		return -1;
	    }

	  line[lptr++] = map[c & 0x3f];

	  state86 = 0;
	  break;
	}
    }

  return 0;
}

/*
 * NAME:	rleflush()
 * DESCRIPTION:	run-length encode data
 */
static
int rleflush(void)
{
  char rle[] = { 0x90, 0x00, 0x90, 0x00 };

  if ((lastch != 0x90 && runlen < 4) ||
      (lastch == 0x90 && runlen < 3))
    {
      /* self representation */

      if (lastch == 0x90)
	{
	  while (runlen--)
	    if (addchars(rle, 2) < 0)
	      return -1;
	}
      else
	{
	  while (runlen--)
	    if (addchars(&lastch, 1) < 0)
	      return -1;
	}
    }
  else
    {
      /* run-length encoded */

      if (lastch == 0x90)
	{
	  rle[3] = runlen;

	  if (addchars(rle, 4) < 0)
	    return -1;
	}
      else
	{
	  rle[1] = lastch;
	  rle[3] = runlen;

	  if (addchars(&rle[1], 3) < 0)
	    return -1;
	}
    }

  runlen = 0;

  return 0;
}

/*
 * NAME:	bh_insert()
 * DESCRIPTION:	encode bytes of data, buffering lines and flushing
 */
int bh_insert(register char *data, register int len)
{
  crc = Hfs_BlockCRC(data, len, crc);

  for ( ; len--; ++data)
    {
      if (runlen)
	{
	  if (runlen == 0xff || lastch != *data)
	    {
	      if (rleflush() < 0)
		return -1;
	    }

	  if (lastch == *data)
	    {
	      ++runlen;
	      continue;
	    }
	}

      lastch = *data;
      runlen = 1;
    }

  return 0;
}

/*
 * NAME:	bh_insertcrc()
 * DESCRIPTION:	insert a two-byte CRC checksum
 */
int bh_insertcrc(void)
{
  short word;

  crc = Hfs_BlockCRC("\0\0", 2, crc);

  word = htons(crc);
  if (bh_insert((char *) &word, 2) < 0)
    return -1;

  crc = 0x0000;

  return 0;
}

/*
 * NAME:	bh_end()
 * DESCRIPTION:	finish BinHex encoding
 */
int bh_end(void)
{
  if (runlen)
    {
      if (rleflush() < 0)
	return -1;
    }

  if (state86)
    {
      if (addchars("\0", 1) < 0)
	return -1;
    }

  if (lptr == sizeof(line) - 1)
    {
      if (flushline() < 0)
	return -1;
    }

  line[lptr++] = ':';

  if (flushline() < 0)
    return -1;

  return 0;
}
